# varray.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program demonstrates vertex arrays.

package require tcl3d

set POINTER 1
set INTERLEAVED 2

set DRAWARRAY 1
set ARRAYELEMENT  2
set DRAWELEMENTS 3

set setupMethod $POINTER
set derefMethod $DRAWARRAY

set vertexVec [tcl3dVectorFromArgs GLint  25  25  100 325  175  25 \
                                         175 325  250  25  325 325]

set colorVec [tcl3dVectorFromArgs GLfloat \
                1.0 0.2 0.2 \
                0.2 0.2 1.0 \
                0.8 1.0 0.2 \
                0.75 0.75 0.75 \
                0.35 0.35 0.35 \
                0.5 0.5 0.5]

set intertwined [tcl3dVectorFromArgs GLfloat \
   1.0 0.2 1.0 100.0 100.0 0.0 \
   1.0 0.2 0.2   0.0 200.0 0.0 \
   1.0 1.0 0.2 100.0 300.0 0.0 \
   0.2 1.0 0.2 200.0 300.0 0.0 \
   0.2 1.0 1.0 300.0 200.0 0.0 \
   0.2 0.2 1.0 200.0 100.0 0.0]

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc setupPointers {} {
    glEnableClientState GL_VERTEX_ARRAY
    glEnableClientState GL_COLOR_ARRAY
 
    glVertexPointer 2 GL_INT 0 $::vertexVec
    glColorPointer  3 GL_FLOAT 0 $::colorVec
}

proc setupInterleave {} {
    glInterleavedArrays GL_C3F_V3F 0 $::intertwined
}

proc CreateCallback { toglwin } {
   glClearColor 0.0 0.0 0.0 0.0
   glShadeModel GL_SMOOTH
   setupPointers
}

proc DisplayCallback { toglwin } {
    glClear GL_COLOR_BUFFER_BIT

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    if { $::derefMethod == $::DRAWARRAY} {
        glDrawArrays GL_TRIANGLES 0 6
    } elseif { $::derefMethod == $::ARRAYELEMENT} {
        glBegin GL_TRIANGLES
        glArrayElement 2
        glArrayElement 3
        glArrayElement 5
        glEnd
   } elseif { $::derefMethod == $::DRAWELEMENTS} {
      set indices [tcl3dVectorFromArgs GLuint 0 1 3 4]

      glDrawElements GL_POLYGON 4 GL_UNSIGNED_INT $indices
      $indices delete
   }
   glFlush
   $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluOrtho2D 0.0 $w 0.0 $h
}

proc ToggleSetupMethod {} {
    if { $::setupMethod == $::POINTER } {
        set ::setupMethod $::INTERLEAVED
        setupInterleave
    } elseif { $::setupMethod == $::INTERLEAVED } {
        set ::setupMethod $::POINTER
        setupPointers
    }
    .fr.toglwin postredisplay
}

proc ToggleDerefMethod {} {
    if { $::derefMethod == $::DRAWARRAY } {
        set ::derefMethod $::ARRAYELEMENT
    } elseif { $::derefMethod == $::ARRAYELEMENT } {
        set ::derefMethod $::DRAWELEMENTS
    } elseif { $::derefMethod == $::DRAWELEMENTS } {
        set ::derefMethod $::DRAWARRAY
    }
    .fr.toglwin postredisplay
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 400 -height 400 -double true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 3
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example varray"

bind . <Key-Escape> "exit"

bind .fr.toglwin <1> "ToggleSetupMethod"
bind .fr.toglwin <2> "ToggleDerefMethod"
bind .fr.toglwin <3> "ToggleDerefMethod"
bind .fr.toglwin <Control-Button-1> "ToggleDerefMethod"

.fr.usage insert end "Mouse-L    Toggle setup method"
.fr.usage insert end "Mouse-MR   Toggle deref method"
.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
