# texturesurf.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program uses evaluators to generate a curved
# surface and automatically generated texture coordinates.

package require tcl3d

set ctrlpoints {
   {{ -1.5 -1.5 4.0} { -0.5 -1.5 2.0} 
    {0.5 -1.5 -1.0} {1.5 -1.5 2.0}} 
   {{ -1.5 -0.5 1.0} { -0.5 -0.5 3.0} 
    {0.5 -0.5 0.0} {1.5 -0.5 -1.0}} 
   {{ -1.5 0.5 4.0} { -0.5 0.5 0.0} 
    {0.5 0.5 3.0} {1.5 0.5 4.0}} 
   {{ -1.5 1.5 -2.0} { -0.5 1.5 -2.0} 
    {0.5 1.5 0.0} {1.5 1.5 -1.0}}
}

set texpts {{{0.0 0.0} {0.0 1.0}}
            {{1.0 0.0} {1.0 1.0}}}

set imageWidth  64
set imageHeight 64

set imgVec [tcl3dVector GLubyte [expr 3*$imageWidth*$imageHeight]]

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc makeImage {} {
    set TWO_PI [expr {2.0*3.14159265}]

    for { set i 0 } { $i < $::imageWidth } { incr i } {
        set ti [expr {$TWO_PI * $i / $::imageWidth}]
        for { set j 0 } { $j < $::imageHeight } { incr j } {
            set tj [expr {$TWO_PI * $j / $::imageHeight}]
            set ind [expr {3 * ($::imageHeight*$i + $j)}]
            $::imgVec set [expr {$ind+0}] [expr {int (127*(1.0+sin($ti)))}]
            $::imgVec set [expr {$ind+1}] [expr {int (127*(1.0+cos(2*$tj)))}]
            $::imgVec set [expr {$ind+2}] [expr {int (127*(1.0+cos($ti+$tj)))}]
        }
    }
}


proc DisplayCallback { toglwin } {
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glColor3f 1.0 1.0 1.0
    glEvalMesh2 GL_FILL 0 20 0 20
    glFlush
    $toglwin swapbuffers
}

proc CreateCallback { toglwin } {
    glMap2f GL_MAP2_VERTEX_3 0 1 3 4 \
            0 1 12 4 [join [join $::ctrlpoints]]
    glMap2f GL_MAP2_TEXTURE_COORD_2 0 1 2 2 \
            0 1 4 2 [join [join $::texpts]]
    glEnable GL_MAP2_TEXTURE_COORD_2
    glEnable GL_MAP2_VERTEX_3
    glMapGrid2f 20 0.0 1.0 20 0.0 1.0
    makeImage
    glTexEnvf GL_TEXTURE_ENV GL_TEXTURE_ENV_MODE $::GL_DECAL
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_WRAP_S $::GL_REPEAT
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_WRAP_T $::GL_REPEAT
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MAG_FILTER $::GL_NEAREST
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MIN_FILTER $::GL_NEAREST
    glTexImage2D GL_TEXTURE_2D 0 $::GL_RGB $::imageWidth $::imageHeight 0 \
                 GL_RGB GL_UNSIGNED_BYTE $::imgVec
    glEnable GL_TEXTURE_2D
    glEnable GL_DEPTH_TEST
    glShadeModel GL_FLAT
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    if { $w <= $h } {
       glOrtho -4.0 4.0 \
                [expr -4.0*double($h)/double($w)] \
                [expr  4.0*double($h)/double($w)] \
                -4.0 4.0
    } else {
        glOrtho [expr -4.0*double($w)/double($h)] \
                [expr  4.0*double($w)/double($h)] \
                -4.0 4.0 -4.0 4.0
    }
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
    glRotatef 85.0 1.0 1.0 1.0
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 500 -height 500 -double true -depth true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 1
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example texturesurf"

bind . <Key-Escape> "exit"

.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
