# texture3d.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program demonstrates using a three-dimensional texture.
# It creates a 3D texture and then renders two rectangles
# with different texture coordinates to obtain different
# "slices" of the 3D texture.

package require tcl3d

set iWidth  16
set iHeight 16
set iDepth  16

set image [tcl3dVector GLubyte [expr $iDepth*$iHeight*$iWidth*3]]

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

#  Create a 16x16x16x3 array with different color values in
#  each array element [r, g, b].  Values range from 0 to 255.
#

proc makeImage {} {
    for { set s 0 } { $s < 16 } { incr s } {
        for { set t 0 } { $t < 16 } { incr t } {
            for { set r 0 } { $r < 16 } { incr r } {
                set ind [expr {((($r*$::iDepth + $t) * $::iHeight) +$s)*3}]
                $::image set [expr {$ind + 0}] [expr {$s * 17}]
                $::image set [expr {$ind + 1}] [expr {$t * 17}]
                $::image set [expr {$ind + 2}] [expr {$r * 17}]
            }
        }
    }
}

proc CreateCallback { toglwin } {
    glClearColor 0.0 0.0 0.0 0.0
    glShadeModel GL_FLAT
    glEnable GL_DEPTH_TEST
 
    makeImage
    glPixelStorei GL_UNPACK_ALIGNMENT 1
 
    set texName [tcl3dVector GLuint 1]
    glGenTextures 1 $texName
    glBindTexture GL_TEXTURE_3D [$texName get 0]
    glTexParameteri GL_TEXTURE_3D GL_TEXTURE_WRAP_S $::GL_CLAMP
    glTexParameteri GL_TEXTURE_3D GL_TEXTURE_WRAP_T $::GL_CLAMP
    glTexParameteri GL_TEXTURE_3D GL_TEXTURE_WRAP_R $::GL_CLAMP
    glTexParameteri GL_TEXTURE_3D GL_TEXTURE_MAG_FILTER $::GL_NEAREST
    glTexParameteri GL_TEXTURE_3D GL_TEXTURE_MIN_FILTER $::GL_NEAREST
    glTexImage3D GL_TEXTURE_3D 0 $::GL_RGB $::iWidth $::iHeight \
                 $::iDepth 0 GL_RGB GL_UNSIGNED_BYTE $::image
    glEnable GL_TEXTURE_3D
}

proc DisplayCallback { toglwin } {
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glBegin GL_QUADS
    glTexCoord3f 0.0 0.0 0.0 ; glVertex3f -2.25 -1.0 0.0
    glTexCoord3f 0.0 1.0 0.0 ; glVertex3f -2.25  1.0 0.0
    glTexCoord3f 1.0 1.0 1.0 ; glVertex3f -0.25  1.0 0.0
    glTexCoord3f 1.0 0.0 1.0 ; glVertex3f -0.25 -1.0 0.0

    glTexCoord3f 0.0 0.0 1.0 ; glVertex3f 0.25 -1.0 0.0
    glTexCoord3f 0.0 1.0 1.0 ; glVertex3f 0.25  1.0 0.0
    glTexCoord3f 1.0 1.0 0.0 ; glVertex3f 2.25  1.0 0.0
    glTexCoord3f 1.0 0.0 0.0 ; glVertex3f 2.25 -1.0 0.0
    glEnd
    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluPerspective 60.0 [expr double($w)/double($h)] 1.0 30.0
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
    glTranslatef 0.0 0.0 -4.0
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 400 -height 400 -double true -depth true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 1
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example texture3d"

bind . <Key-Escape> "exit"

.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]

if { ! [tcl3dOglHaveVersion 1 2] } {
    tk_messageBox -icon warning -type ok -title "Invalid OpenGL version" \
                  -message [format "Feature needs OpenGL >= 1.2. Only have %s" \
                            [glGetString GL_VERSION]]
}
