# surface.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program draws a NURBS surface in the shape of a 
# symmetrical hill. The 'c' keyboard key allows you to 
# toggle the visibility of the control points themselves.  
# Note that some of the control points are hidden by the  
# surface itself.

package require tcl3d

set showPoints false

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

#
# Initializes the control points of the surface to a small hill.
# The control points range from -3 to +3 in x, y, and z
#
proc init_surface {} {
    global ctlpoints

    for { set u 0 } { $u < 4 } { incr u } {
        for { set v 0 } { $v < 4 } { incr v } {
            lappend ctlpoints { 0.0 0.0 0.0 }
        }
    }
    for { set u 0 } { $u < 4 } { incr u } {
        for { set v 0 } { $v < 4 } { incr v } {
            lset ctlpoints [expr $u * 4 + $v] 0 [expr 2.0*(double($u) - 1.5)]
            lset ctlpoints [expr $u * 4 + $v] 1 [expr 2.0*(double($v) - 1.5)]
            if { ($u == 1 || $u == 2) && ($v == 1 || $v == 2) } {
                lset ctlpoints [expr $u *4 + $v] 2 3.0
            } else {
                lset ctlpoints [expr $u *4 + $v] 2 -3.0
            }
        }
    }
}

# Initialize material property and depth buffer.
#
proc CreateCallback { toglwin } {
    global theNurb

    set mat_diffuse { 0.7 0.7 0.7 1.0 }
    set mat_specular { 1.0 1.0 1.0 1.0 }
    set mat_shininess { 100.0 }

    glClearColor 0.0 0.0 0.0 0.0
    glMaterialfv GL_FRONT GL_DIFFUSE   $mat_diffuse
    glMaterialfv GL_FRONT GL_SPECULAR  $mat_specular
    glMaterialfv GL_FRONT GL_SHININESS $mat_shininess

    glEnable GL_LIGHTING
    glEnable GL_LIGHT0
    glEnable GL_DEPTH_TEST
    glEnable GL_AUTO_NORMAL
    glEnable GL_NORMALIZE

    init_surface

    set theNurb [gluNewNurbsRenderer]
    gluNurbsProperty $theNurb GLU_SAMPLING_TOLERANCE 25.0
    gluNurbsProperty $theNurb GLU_DISPLAY_MODE $::GLU_FILL
}

proc DisplayCallback { toglwin } {
    global theNurb
    global showPoints
    global ctlpoints

    set knots {0.0 0.0 0.0 0.0 1.0 1.0 1.0 1.0}

    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glPushMatrix
    glRotatef 330.0 1.0 0.0 0.0
    glScalef  0.5 0.5 0.5

    gluBeginSurface $theNurb
    gluNurbsSurface $theNurb 8 $knots 8 $knots \
                    [expr 4 * 3] 3 [join $ctlpoints] \
                    4 4 GL_MAP2_VERTEX_3
    gluEndSurface $theNurb

    if { $showPoints } {
        glPointSize 5.0
        glDisable GL_LIGHTING
        glColor3f 1.0 1.0 0.0
        glBegin GL_POINTS
        for { set i 0 } { $i < 4 } { incr i } {
            for { set j 0 } { $j < 4 } { incr j } {
                set pntList [lindex $ctlpoints [expr $i*4 + $j]]
                glVertex3f [lindex $pntList 0] \
                           [lindex $pntList 1] \
                           [lindex $pntList 2]
            }
        }
        glEnd
        glEnable GL_LIGHTING
    }
    glPopMatrix
    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluPerspective 45.0 [expr double($w)/double($h)] 3.0 8.0

    glMatrixMode GL_MODELVIEW
    glLoadIdentity
    glTranslatef 0.0 0.0 -5.0
}

proc ToggleShowPoints {} {
    global showPoints

    set showPoints [expr ! $showPoints]
    .fr.toglwin postredisplay
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 500 -height 500 -double true -depth true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 2
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example surface"

bind . <Key-c> "ToggleShowPoints"
bind . <Key-Escape> "exit"

.fr.usage insert end "Key-c      Toggle control points"
.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
