# stroke.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program demonstrates some characters of a 
# stroke (vector) font.  The characters are represented
# by display lists, which are given numbers which 
# correspond to the ASCII values of the characters.
# Use of glCallLists() is demonstrated.

package require tcl3d

set X 0
set Y 1
set TYPE 2

set Adata {
   {0 0 PT} {0 9 PT} {1 10 PT} {4 10 PT} 
   {5 9 PT} {5 0 STROKE} {0 5 PT} {5 5 END}
}

set Edata {
   {5 0 PT} {0 0 PT} {0 10 PT} {5 10 STROKE}
   {0 5 PT} {4 5 END}
}

set Pdata {
   {0 0 PT} {0 10 PT} {4 10 PT} {5 9 PT} {5 6 PT} 
   {4 5 PT} {0 5 END}
}

set Rdata {
   {0 0 PT} {0 10 PT}  {4 10 PT} {5 9 PT} {5 6 PT} 
   {4 5 PT} {0 5 STROKE} {3 5 PT} {5 0 END}
}

set Sdata {
   {0 1 PT} {1 0 PT} {4 0 PT} {5 1 PT} {5 4 PT}
   {4 5 PT} {1 5 PT} {0 6 PT} {0 9 PT} {1 10 PT}
   {4 10 PT} {5 9 END}
}

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# drawLetter interprets the instructions from the array
# for that letter and renders the letter with line segments.
#
proc drawLetter { letter } {
    global X Y TYPE
    global PT STROKE END

    glBegin GL_LINE_STRIP
    foreach l $letter {
        switch -exact [lindex $l $TYPE] {
            PT {
                glVertex2f [lindex $l 0] [lindex $l 1]
            }
            STROKE {
                glVertex2f [lindex $l 0] [lindex $l 1]
                glEnd
                glBegin GL_LINE_STRIP
            }
            END {
                glVertex2f [lindex $l 0] [lindex $l 1]
                glEnd
                glTranslatef 8.0 0.0 0.0
                return
            }
        }
    }
}

# Create a display list for each of 6 characters
proc CreateCallback { toglwin } {
    glShadeModel GL_FLAT

    set base [glGenLists 128]
    glListBase $base

    glNewList [expr $base + [tcl3dCharToNum "A"]] GL_COMPILE
        drawLetter $::Adata
    glEndList
    glNewList [expr $base + [tcl3dCharToNum "E"]] GL_COMPILE
        drawLetter $::Edata
    glEndList
    glNewList [expr $base + [tcl3dCharToNum "P"]] GL_COMPILE
        drawLetter $::Pdata
    glEndList
    glNewList [expr $base + [tcl3dCharToNum "R"]] GL_COMPILE
        drawLetter $::Rdata
    glEndList
    glNewList [expr $base + [tcl3dCharToNum "S"]] GL_COMPILE
        drawLetter $::Sdata
    glEndList
    glNewList [expr $base + [tcl3dCharToNum " "]] GL_COMPILE
        glTranslatef 8.0 0.0 0.0
    glEndList
}

set test1 "A SPARE SERAPE APPEARS AS"
set test2 "APES PREPARE RARE PEPPERS"

proc printStrokedString { s } {
    set len [string length $s]
    if { $len > 0 } {
        set sa [tcl3dVectorFromString GLubyte $s]
        glCallLists $len GL_BYTE $sa
        $sa delete
    }
}

proc DisplayCallback { toglwin } {
    global test1 test2

    glClear GL_COLOR_BUFFER_BIT

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glColor3f 1.0 1.0 1.0
    glPushMatrix
    glScalef 2.0 2.0 2.0
    glTranslatef 10.0 30.0 0.0
    printStrokedString $test1
    glPopMatrix
    glPushMatrix
    glScalef 2.0 2.0 2.0
    glTranslatef 10.0 13.0 0.0
    printStrokedString $test2
    glPopMatrix
    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluOrtho2D 0.0 $w 0.0 $h
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 440 -height 120 -double true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 1
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example stroke"

bind . <Key-Escape> "exit"

.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
