# scene.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program demonstrates the use of the GL lighting model.
# Objects are drawn using a grey material characteristic. 
# A single light source illuminates the objects.

package require tcl3d

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# Initialize material property and light source.

proc CreateCallback { toglwin } {
    set light_ambient  { 0.0 0.0 0.0 1.0 }
    set light_diffuse  { 1.0 1.0 1.0 1.0 }
    set light_specular { 1.0 1.0 1.0 1.0 }
    # light_position is NOT default value
    set light_position { 1.0 1.0 1.0 0.0 }

    glLightfv GL_LIGHT0 GL_AMBIENT  $light_ambient
    glLightfv GL_LIGHT0 GL_DIFFUSE  $light_diffuse
    glLightfv GL_LIGHT0 GL_SPECULAR $light_specular
    glLightfv GL_LIGHT0 GL_POSITION $light_position
    
    glEnable GL_LIGHTING
    glEnable GL_LIGHT0
    glEnable GL_DEPTH_TEST
}

proc DisplayCallback { toglwin } {
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glPushMatrix
    glRotatef 20.0 1.0 0.0 0.0

    glPushMatrix
    glTranslatef -0.75 0.5 0.0 
    glRotatef 90.0 1.0 0.0 0.0
    glutSolidTorus 0.275 0.85 15 15
    glPopMatrix

    glPushMatrix
    glTranslatef -0.75 -0.5 0.0
    glRotatef 270.0 1.0 0.0 0.0
    glutSolidCone 1.0 2.0 15 15
    glPopMatrix

    glPushMatrix
    glTranslatef 0.75 0.0 -1.0
    glutSolidSphere 1.0 15 15
    glPopMatrix

    glPopMatrix
    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    if { $w <= $h } {
        glOrtho -2.5 2.5 [expr -2.5*double($h)/double($w)] \
                [expr 2.5*double($h)/double($w)] -10.0 10.0
    } else {
        glOrtho [expr -2.5*double($w)/double($h)] \
                [expr  2.5*double($w)/double($h)] -2.5 2.5 -10.0 10.0
    }
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 500 -height 500 -double true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 1
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example scene"

bind . <Key-Escape> "exit"

.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
