# quadric.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program demonstrates the use of some of the gluQuadric*
# routines. Quadric objects are created with some quadric
# properties and the callback routine to handle errors.
# Note that the cylinder has no top or bottom and the circle
# has a hole in it.

package require tcl3d

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc CreateCallback { toglwin } {
    set mat_ambient { 0.5 0.5 0.5 1.0 }
    set mat_specular { 1.0 1.0 1.0 1.0 }
    set mat_shininess { 50.0 }
    set light_position { 1.0 1.0 1.0 0.0 }
    set model_ambient { 0.5 0.5 0.5 1.0 }
 
    glClearColor 0.0 0.0 0.0 0.0
 
    glMaterialfv GL_FRONT GL_AMBIENT $mat_ambient
    glMaterialfv GL_FRONT GL_SPECULAR $mat_specular
    glMaterialfv GL_FRONT GL_SHININESS $mat_shininess
    glLightfv GL_LIGHT0 GL_POSITION $light_position
    glLightModelfv GL_LIGHT_MODEL_AMBIENT $model_ambient
 
    glEnable GL_LIGHTING
    glEnable GL_LIGHT0
    glEnable GL_DEPTH_TEST
 
    #  Create 4 display lists, each with a different quadric object.
    #  Different drawing styles and surface normal specifications
    #  are demonstrated.
    #
    set ::startList [glGenLists 4]
    set qobj  [gluNewQuadric]

    gluQuadricDrawStyle $qobj GLU_FILL ; # smooth shaded
    gluQuadricNormals $qobj GLU_SMOOTH
    glNewList $::startList GL_COMPILE
    gluSphere $qobj 0.75 15 10
    glEndList

    gluQuadricDrawStyle $qobj GLU_FILL ; # flat shaded
    gluQuadricNormals $qobj GLU_FLAT
    glNewList [expr $::startList+1] GL_COMPILE
    gluCylinder $qobj 0.5 0.3 1.0 15 5
    glEndList

    gluQuadricDrawStyle $qobj GLU_LINE ; # all polygons wireframe
    gluQuadricNormals $qobj GLU_NONE
    glNewList [expr $::startList+2] GL_COMPILE
    gluDisk $qobj 0.25 1.0 20 4
    glEndList

    gluQuadricDrawStyle $qobj GLU_SILHOUETTE ; # boundary only
    gluQuadricNormals $qobj GLU_NONE
    glNewList [expr $::startList+3] GL_COMPILE
    gluPartialDisk $qobj 0.0 1.0 20 4 0.0 225.0
    glEndList
}

proc DisplayCallback { toglwin } {
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glPushMatrix

    glEnable GL_LIGHTING
    glShadeModel GL_SMOOTH
    glTranslatef -1.0 -1.0 0.0
    glCallList $::startList

    glShadeModel GL_FLAT
    glTranslatef 0.0 2.0 0.0
    glPushMatrix
    glRotatef 300.0 1.0 0.0 0.0
    glCallList [expr $::startList+1]
    glPopMatrix

    glDisable GL_LIGHTING
    glColor3f 0.0 1.0 1.0
    glTranslatef 2.0 -2.0 0.0
    glCallList [expr $::startList+2]

    glColor3f 1.0 1.0 0.0
    glTranslatef 0.0 2.0 0.0
    glCallList [expr $::startList+3]

    glPopMatrix
    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    if { $w <= $h } {
       glOrtho -2.5 2.5 \
               [expr -2.5*double($h)/double($w)] [expr 2.5*double($h)/double($w)] \
               -10.0 10.0
    } else {
       glOrtho [expr -2.5*double($w)/double($h)] \
               [expr  2.5*double($w)/double($h)] \
               -2.5 2.5 -10.0 10.0
    }
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 500 -height 500 -double true -depth true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 1
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example quadric"

bind . <Key-Escape> "exit"

.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
