# polyoff.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program demonstrates polygon offset to draw a shaded
# polygon and its wireframe counterpart without ugly visual
# artifacts ("stitching").

package require tcl3d

set spinx  0
set spiny  0
set tdist  0.0
set polyfactor 1.0
set polyunits  1.0

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# display() draws two spheres, one with a gray, diffuse material,
# the other sphere with a magenta material with a specular highlight.
#
proc DisplayCallback { toglwin } {
    set mat_ambient  { 0.8 0.8 0.8 1.0 }
    set mat_diffuse  { 1.0 0.0 0.5 1.0 }
    set mat_specular { 1.0 1.0 1.0 1.0 }
    set gray  { 0.8 0.8 0.8 1.0 }
    set black { 0.0 0.0 0.0 1.0 }

    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glPushMatrix
    glTranslatef 0.0 0.0 $::tdist
    glRotatef $::spinx 1.0 0.0 0.0
    glRotatef $::spiny 0.0 1.0 0.0

    glMaterialfv GL_FRONT GL_AMBIENT_AND_DIFFUSE $gray
    glMaterialfv GL_FRONT GL_SPECULAR $black
    glMaterialf GL_FRONT GL_SHININESS 0.0
    glEnable GL_LIGHTING
    glEnable GL_LIGHT0
    glEnable GL_POLYGON_OFFSET_FILL
    glPolygonOffset $::polyfactor $::polyunits
    glCallList $::list
    glDisable GL_POLYGON_OFFSET_FILL

    glDisable GL_LIGHTING
    glDisable GL_LIGHT0
    glColor3f  1.0 1.0 1.0
    glPolygonMode GL_FRONT_AND_BACK GL_LINE
    glCallList $::list
    glPolygonMode GL_FRONT_AND_BACK GL_FILL

    glPopMatrix
    glFlush
    $toglwin swapbuffers
}

# specify initial properties
# create display list with sphere  
# initialize lighting and depth buffer
#
proc CreateCallback { toglwin } {
    set light_ambient  { 0.0 0.0 0.0 1.0 }
    set light_diffuse  { 1.0 1.0 1.0 1.0 }
    set light_specular { 1.0 1.0 1.0 1.0 }
    set light_position { 1.0 1.0 1.0 0.0 }

    set global_ambient { 0.2 0.2 0.2 1.0 }

    glClearColor 0.0 0.0 0.0 1.0

    set ::list [glGenLists 1]
    glNewList $::list GL_COMPILE
        glutSolidSphere 1.0 20 12
    glEndList

    glEnable GL_DEPTH_TEST

    glLightfv GL_LIGHT0 GL_AMBIENT $light_ambient
    glLightfv GL_LIGHT0 GL_DIFFUSE $light_diffuse
    glLightfv GL_LIGHT0 GL_SPECULAR $light_specular
    glLightfv GL_LIGHT0 GL_POSITION $light_position
    glLightModelfv GL_LIGHT_MODEL_AMBIENT $global_ambient
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluPerspective 45.0 [expr double($w)/double($h)] 1.0 10.0
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
    gluLookAt 0.0 0.0 5.0 0.0 0.0 0.0 0.0 1.0 0.0
}

proc UpdateMsg { str } {
    .fr.usage configure -state normal
    .fr.usage delete end
    .fr.usage insert end $str
    .fr.usage configure -state disabled
}

proc IncrPolyUnits {} {
    set ::polyunits [expr $::polyunits + 1.0]
    UpdateMsg "polyunits is $::polyunits"
    .fr.toglwin postredisplay
}

proc DecrPolyUnits {} {
    set ::polyunits [expr $::polyunits - 1.0]
    UpdateMsg "polyunits is $::polyunits"
    .fr.toglwin postredisplay
}

proc IncrPolyFactors {} {
    set ::polyfactor [expr $::polyfactor + 0.1]
    UpdateMsg "polyfactor is $::polyfactor"
    .fr.toglwin postredisplay
}

proc DecrPolyFactors {} {
    set ::polyfactor [expr $::polyfactor - 0.1]
    UpdateMsg "polyfactor is $::polyfactor"
    .fr.toglwin postredisplay
}

proc DecrDistance {} {
    if { $::tdist < 4.0 } {
        set ::tdist [expr $::tdist + 0.5]
        .fr.toglwin postredisplay
    }
}

proc IncrDistance {} {
    if { $::tdist > -5.0 } {
        set ::tdist [expr $::tdist - 0.5]
        .fr.toglwin postredisplay
    }
}

proc IncrSpinX {} {
    set ::spinx [expr ($::spinx + 5) % 360]
    .fr.toglwin postredisplay
}

proc IncrSpinY {} {
    set ::spiny [expr ($::spiny + 5) % 360]
    .fr.toglwin postredisplay
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -double true -depth true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 10
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example polyoff"

bind . <Key-t> "DecrDistance"
bind . <Key-T> "IncrDistance"
bind . <Key-f> "DecrPolyFactors"
bind . <Key-F> "IncrPolyFactors"
bind . <Key-u> "DecrPolyUnits"
bind . <Key-U> "IncrPolyUnits"
bind . <1> "IncrSpinX"
bind . <2> "IncrSpinY"
bind . <3> "IncrSpinY"
bind . <Control-Button-1> "IncrSpinY"
bind . <Key-Escape> "exit"

.fr.usage insert end "Key-t      DecrDistance"
.fr.usage insert end "Key-T      IncrDistance"
.fr.usage insert end "Key-f      DecrPolyFactors"
.fr.usage insert end "Key-F      IncrPolyFactors"
.fr.usage insert end "Key-u      DecrPolyUnits"
.fr.usage insert end "Key-U      IncrPolyUnits"
.fr.usage insert end "Mouse-L    IncrSpinX"
.fr.usage insert end "Mouse-MR   IncrSpinY"
.fr.usage insert end "Key-Escape Exit"
.fr.usage insert end "Initialized"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
