# picksquare.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# Use of multiple names and picking are demonstrated.  
# A 3x3 grid of squares is drawn.  When the left mouse 
# button is pressed, all squares under the cursor position 
# have their color changed.

package require tcl3d

tcl3dConsoleCreate .tcl3dOutputConsole "# " "Picksquare Output"

set BUFSIZE 512
set selectBuffer [tcl3dVector GLuint $::BUFSIZE]

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# Clear color value for every square on the board
proc CreateCallback { toglwin } {
    set ::board {}
    for { set i 0 } { $i < 3 } { incr i } {
        for { set j 0 } { $j < 3 } { incr j } {
            lappend ::board 0
        }
    }
    glClearColor 0.0 0.0 0.0 0.0
}

# The nine squares are drawn.  In selection mode, each 
# square is given two names:  one for the row and the 
# other for the column on the grid.  The color of each 
# square is determined by its position on the grid, and 
# the value in the board[][] array.
#
proc drawSquares { mode } {
    for { set i 0 } { $i < 3 } { incr i } {
        if { $mode == $::GL_SELECT } {
            glLoadName $i 
        }
        for { set j 0 } { $j < 3 } { incr j } {
            if { $mode == $::GL_SELECT } {
                glPushName $j
            }
            glColor3f [expr $i/3.0] [expr $j/3.0] \
                      [expr [lindex $::board [expr $i*3 + $j]]/3.0]
            glRecti $i $j [expr $i+1] [expr $j+1]
            if { $mode == $::GL_SELECT } {
                glPopName
            }
        }
    }
}

# processHits prints out the contents of the 
# selection array.
#
proc processHits { hits } {
    set count 0
    puts  "hits = $hits"
    for { set i 0 } { $i < $hits } { incr i } {
        set names [$::selectBuffer get $count]
        puts " number of names for hit = $names"
        incr count
        puts -nonewline [format "  z1 is %g;"  \
               [expr double ([$::selectBuffer get $count]) / 0x7fffffff]]
        incr count
        puts [format " z2 is %g" \
               [expr double ([$::selectBuffer get $count]) / 0x7fffffff]]
        incr count
        puts -nonewline "   names are "
        for { set j 0 } { $j < $names } { incr j } {
            set val [$::selectBuffer get $count]
            puts -nonewline [format "%d " $val]
            if { $j == 0 } { 
                set ii $val
            } elseif { $j == 1 } { 
                set jj $val
            }
            incr count
        }
        puts ""
        set bval [expr ([lindex $::board [expr $ii*3 + $jj]] + 1) % 3]
        lset ::board [expr $ii*3 + $jj] $bval
    }
}

# pickSquares() sets up selection mode, name stack, 
# and projection matrix for picking.  Then the 
# objects are drawn.
#

proc pickSquares { x y } {
    set viewport [tcl3dVector GLint 4]
    glGetIntegerv GL_VIEWPORT $viewport

    glSelectBuffer $::BUFSIZE $::selectBuffer
    glRenderMode GL_SELECT

    glInitNames
    glPushName 0
 
    glMatrixMode GL_PROJECTION
    glPushMatrix
    glLoadIdentity
    # create 5x5 pixel picking region near cursor location
    gluPickMatrix $x [expr [$viewport get 3] - $y] 5.0 5.0 $viewport
    gluOrtho2D 0.0 3.0 0.0 3.0
    drawSquares $::GL_SELECT

    glMatrixMode GL_PROJECTION
    glPopMatrix
    glFlush

    set hits [glRenderMode GL_RENDER]
    processHits $hits
    $viewport delete
    .fr.toglwin postredisplay
} 

proc DisplayCallback { toglwin } {
    glClear GL_COLOR_BUFFER_BIT

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    drawSquares $::GL_RENDER
    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluOrtho2D 0.0 3.0 0.0 3.0
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 400 -height 400 -double true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 2
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example picksquare"

bind . <Key-Escape> "exit"

bind .fr.toglwin <1> "pickSquares %x %y"

.fr.usage insert end "Key-Escape Exit"
.fr.usage insert end "Mouse-L    Get pick results"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
