# mvarray.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program demonstrates multiple vertex arrays,
# specifically the OpenGL routine glMultiDrawElements().

package require tcl3d

set oneIndices [tcl3dVectorFromArgs GLubyte 0 1 2 3 4 5 6]
set twoIndices [tcl3dVectorFromArgs GLubyte 1 7 8 9 10 11]
set count [tcl3dVectorFromArgs GLuint 7 6]
# glMultiDrawElements is curently implemented as a Tcl utility 
# function, because void ** are currently not supported by Tcl3D wrapping.
set indices [list $oneIndices $twoIndices]

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc setupPointer {} {
    set vertices [tcl3dVectorFromArgs GLint \
        25  25  75  75 100 125 \
       150  75 200 175 250 150 \
       300 125 100 200 150 250 \
       200 225 250 300 300 250]

    glEnableClientState GL_VERTEX_ARRAY
    glVertexPointer 2 GL_INT 0 $vertices
}

proc CreateCallback { toglwin } {
    glClearColor 0.0 0.0 0.0 0.0
    glShadeModel GL_SMOOTH
    setupPointer
}

proc DisplayCallback { toglwin } {
    glClear GL_COLOR_BUFFER_BIT

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glColor3f 1.0 1.0 1.0
    glMultiDrawElements GL_LINE_STRIP $::count $::GL_UNSIGNED_BYTE $::indices 2
    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluOrtho2D 0.0 $w 0.0 $h
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 350 -height 350 -double true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 1
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example mvarray"

bind . <Key-Escape> "exit"

.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]

if { ! [tcl3dOglHaveVersion 1 3] } {
    tk_messageBox -icon warning -type ok -title "Invalid OpenGL version" \
                  -message [format "Feature needs OpenGL >= 1.3. Only have %s" \
                            [glGetString GL_VERSION]]
}
