# movelight.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program demonstrates when to issue lighting and
# transformation commands to render a model with a light
# which is moved by a modeling transformation (rotate or
# translate).  The light position is reset after the modeling
# transformation is called.  The eye position does not change.
#
# A sphere is drawn using a grey material characteristic.
# A single light source illuminates the object.
#
# Interaction:  pressing the left mouse button alters
# the modeling transformation (x rotation) by 30 degrees.
# The scene is then redrawn with the light in a new position.

package require tcl3d

set spin 0

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# Initialize material property, light source, lighting model, and depth buffer.
proc CreateCallback { toglwin } {
    glClearColor 0.0 0.0 0.0 0.0
    glShadeModel GL_SMOOTH
    glEnable GL_LIGHTING
    glEnable GL_LIGHT0
    glEnable GL_DEPTH_TEST
}

# Here is where the light position is reset after the modeling
# transformation (glRotated) is called.  This places the
# light at a new position in world coordinates.  The cube
# represents the position of the light.
#
proc DisplayCallback { toglwin } {
    set position { 0.0 0.0 1.5 1.0 }

    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glPushMatrix
    gluLookAt 0.0 0.0 5.0 0.0 0.0 0.0 0.0 1.0 0.0

    glPushMatrix
    glRotated $::spin 1.0 0.0 0.0
    glLightfv GL_LIGHT0 GL_POSITION $position

    glTranslated 0.0 0.0 1.5
    glDisable GL_LIGHTING
    glColor3f 0.0 1.0 1.0
    glutWireCube 0.1
    glEnable GL_LIGHTING
    glPopMatrix

    glutSolidTorus 0.275 0.85 8 15
    glPopMatrix
    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluPerspective 40.0 [expr double($w)/double($h)] 1.0 20.0
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
}

proc AddSpin {} {
    set ::spin [expr ($::spin + 30) % 360]
    .fr.toglwin postredisplay
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 500 -height 500 -double true -depth true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 2
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example movelight"

bind .fr.toglwin <1> "AddSpin"
bind . <Key-Escape> "exit"

.fr.usage insert end "Mouse-L    Move light"
.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
