# mipmap.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program demonstrates using mipmaps for texture maps.
# To overtly show the effect of mipmaps, each mipmap reduction
# level has a solidly colored, contrasting texture image.
# Thus, the quadrilateral which is drawn is drawn with several
# different colors.

package require tcl3d

set mipmapImage32 [tcl3dVector GLubyte [expr 32*32*4]]
set mipmapImage16 [tcl3dVector GLubyte [expr 16*16*4]]
set mipmapImage8  [tcl3dVector GLubyte [expr 8*8*4]]
set mipmapImage4  [tcl3dVector GLubyte [expr 4*4*4]]
set mipmapImage2  [tcl3dVector GLubyte [expr 2*2*4]]
set mipmapImage1  [tcl3dVector GLubyte [expr 1*1*4]]

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc makeImages {} {
    for { set i 0 } { $i < 32 } { incr i } {
        for { set j 0 } { $j < 32 } { incr j } {
            $::mipmapImage32 set [expr $i*32*4 + $j*4 + 0] 255
            $::mipmapImage32 set [expr $i*32*4 + $j*4 + 1] 255
            $::mipmapImage32 set [expr $i*32*4 + $j*4 + 2] 0
            $::mipmapImage32 set [expr $i*32*4 + $j*4 + 3] 255
        }
    }
    for { set i 0 } { $i < 16 } { incr i } {
        for { set j 0 } { $j < 16 } { incr j } {
            $::mipmapImage16 set [expr $i*16*4 + $j*4 + 0] 255
            $::mipmapImage16 set [expr $i*16*4 + $j*4 + 1] 0
            $::mipmapImage16 set [expr $i*16*4 + $j*4 + 2] 255
            $::mipmapImage16 set [expr $i*16*4 + $j*4 + 3] 255
        }
    }
    for { set i 0 } { $i < 8 } { incr i } {
        for { set j 0 } { $j < 8 } { incr j } {
            $::mipmapImage8 set [expr $i*8*4 + $j*4 + 0] 255
            $::mipmapImage8 set [expr $i*8*4 + $j*4 + 1] 0
            $::mipmapImage8 set [expr $i*8*4 + $j*4 + 2] 0
            $::mipmapImage8 set [expr $i*8*4 + $j*4 + 3] 255
        }
    }
    for { set i 0 } { $i < 4 } { incr i } {
        for { set j 0 } { $j < 4 } { incr j } {
            $::mipmapImage4 set [expr $i*4*4 + $j*4 + 0] 0
            $::mipmapImage4 set [expr $i*4*4 + $j*4 + 1] 255
            $::mipmapImage4 set [expr $i*4*4 + $j*4 + 2] 0
            $::mipmapImage4 set [expr $i*4*4 + $j*4 + 3] 255
        }
    }
    for { set i 0 } { $i < 2 } { incr i } {
        for { set j 0 } { $j < 2 } { incr j } {
            $::mipmapImage2 set [expr $i*2*4 + $j*4 + 0] 0
            $::mipmapImage2 set [expr $i*2*4 + $j*4 + 1] 0
            $::mipmapImage2 set [expr $i*2*4 + $j*4 + 2] 255
            $::mipmapImage2 set [expr $i*2*4 + $j*4 + 3] 255
        }
    }
    $::mipmapImage1 set 0 255
    $::mipmapImage1 set 1 255
    $::mipmapImage1 set 2 255
    $::mipmapImage1 set 3 255
}

proc CreateCallback { toglwin } {
    glEnable GL_DEPTH_TEST
    glShadeModel GL_FLAT

    glTranslatef 0.0 0.0 -3.6
    makeImages
    glPixelStorei GL_UNPACK_ALIGNMENT 1

    set ::texName [tcl3dVector GLuint 1]
    glGenTextures 1 $::texName
    glBindTexture GL_TEXTURE_2D [$::texName get 0]
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_WRAP_S $::GL_REPEAT
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_WRAP_T $::GL_REPEAT
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MAG_FILTER $::GL_NEAREST
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MIN_FILTER \
                    $::GL_NEAREST_MIPMAP_NEAREST
    glTexImage2D GL_TEXTURE_2D 0 $::GL_RGBA 32 32 0 \
                 GL_RGBA GL_UNSIGNED_BYTE $::mipmapImage32
    glTexImage2D GL_TEXTURE_2D 1 $::GL_RGBA 16 16 0 \
                 GL_RGBA GL_UNSIGNED_BYTE $::mipmapImage16
    glTexImage2D GL_TEXTURE_2D 2 $::GL_RGBA 8 8 0 \
                 GL_RGBA GL_UNSIGNED_BYTE $::mipmapImage8
    glTexImage2D GL_TEXTURE_2D 3 $::GL_RGBA 4 4 0 \
                 GL_RGBA GL_UNSIGNED_BYTE $::mipmapImage4
    glTexImage2D GL_TEXTURE_2D 4 $::GL_RGBA 2 2 0 \
                 GL_RGBA GL_UNSIGNED_BYTE $::mipmapImage2
    glTexImage2D GL_TEXTURE_2D 5 $::GL_RGBA 1 1 0 \
                 GL_RGBA GL_UNSIGNED_BYTE $::mipmapImage1
 
    glTexEnvf GL_TEXTURE_ENV GL_TEXTURE_ENV_MODE $::GL_DECAL
    glEnable GL_TEXTURE_2D
}

proc DisplayCallback { toglwin } {
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glBindTexture GL_TEXTURE_2D [$::texName get 0]
    glBegin GL_QUADS
    glTexCoord2f 0.0 0.0 ; glVertex3f -2.0 -1.0 0.0
    glTexCoord2f 0.0 8.0 ; glVertex3f -2.0 1.0 0.0
    glTexCoord2f 8.0 8.0 ; glVertex3f 2000.0 1.0 -6000.0
    glTexCoord2f 8.0 0.0 ; glVertex3f 2000.0 -1.0 -6000.0
    glEnd
    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluPerspective 60.0 [expr double($w)/double($h)] 1.0 30000.0
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 500 -height 500 -double true -depth true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 1
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example mipmap"

bind . <Key-Escape> "exit"

.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
