# material.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program demonstrates the use of the GL lighting model.
# Several objects are drawn using different material characteristics.
# A single light source illuminates the objects.

package require tcl3d

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# Initialize z-buffer, projection matrix, light source, 
# and lighting model.  Do not specify a material property here.
#
proc CreateCallback { toglwin } {
    set ambient { 0.0 0.0 0.0 1.0 }
    set diffuse { 1.0 1.0 1.0 1.0 }
    set specular { 1.0 1.0 1.0 1.0 }
    set position { 0.0 3.0 2.0 0.0 }
    set lmodel_ambient { 0.4 0.4 0.4 1.0 }
    set local_view { 0.0 }
 
    glClearColor 0.0 0.1 0.1 0.0
    glEnable GL_DEPTH_TEST
    glShadeModel GL_SMOOTH
 
    glLightfv GL_LIGHT0 GL_AMBIENT $ambient
    glLightfv GL_LIGHT0 GL_DIFFUSE $diffuse
    glLightfv GL_LIGHT0 GL_POSITION $position
    glLightModelfv GL_LIGHT_MODEL_AMBIENT $lmodel_ambient
    glLightModelfv GL_LIGHT_MODEL_LOCAL_VIEWER $local_view
 
    glEnable GL_LIGHTING
    glEnable GL_LIGHT0
}

# Draw twelve spheres in 3 rows with 4 columns.  
# The spheres in the first row have materials with no ambient reflection.
# The second row has materials with significant ambient reflection.
# The third row has materials with colored ambient reflection.
#
# The first column has materials with blue, diffuse reflection only.
# The second column has blue diffuse reflection, as well as specular
# reflection with a low shininess exponent.
# The third column has blue diffuse reflection, as well as specular
# reflection with a high shininess exponent (a more concentrated highlight).
# The fourth column has materials which also include an emissive component.
#
# glTranslatef() is used to move spheres to their appropriate locations.
#

proc DisplayCallback { toglwin } {
    set no_mat { 0.0 0.0 0.0 1.0 }
    set mat_ambient { 0.7 0.7 0.7 1.0 }
    set mat_ambient_color { 0.8 0.8 0.2 1.0 }
    set mat_diffuse { 0.1 0.5 0.8 1.0 }
    set mat_specular { 1.0 1.0 1.0 1.0 }
    set no_shininess { 0.0 }
    set low_shininess { 5.0 }
    set high_shininess { 100.0 }
    set mat_emission {0.3 0.2 0.2 0.0}
 
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    # draw sphere in first row, first column
    # diffuse reflection only; no ambient or specular  
    #
    glPushMatrix
    glTranslatef -3.75 3.0 0.0
    glMaterialfv GL_FRONT GL_AMBIENT   $no_mat
    glMaterialfv GL_FRONT GL_DIFFUSE   $mat_diffuse
    glMaterialfv GL_FRONT GL_SPECULAR  $no_mat
    glMaterialfv GL_FRONT GL_SHININESS $no_shininess
    glMaterialfv GL_FRONT GL_EMISSION  $no_mat
    glutSolidSphere 1.0 16 16
    glPopMatrix

    # draw sphere in first row, second column
    # diffuse and specular reflection; low shininess; no ambient
    #
    glPushMatrix
    glTranslatef -1.25 3.0 0.0
    glMaterialfv GL_FRONT GL_AMBIENT   $no_mat
    glMaterialfv GL_FRONT GL_DIFFUSE   $mat_diffuse
    glMaterialfv GL_FRONT GL_SPECULAR  $mat_specular
    glMaterialfv GL_FRONT GL_SHININESS $low_shininess
    glMaterialfv GL_FRONT GL_EMISSION  $no_mat
    glutSolidSphere 1.0 16 16
    glPopMatrix

    # draw sphere in first row, third column
    # diffuse and specular reflection; high shininess; no ambient
    #
    glPushMatrix
    glTranslatef 1.25 3.0 0.0
    glMaterialfv GL_FRONT GL_AMBIENT   $no_mat
    glMaterialfv GL_FRONT GL_DIFFUSE   $mat_diffuse
    glMaterialfv GL_FRONT GL_SPECULAR  $mat_specular
    glMaterialfv GL_FRONT GL_SHININESS $high_shininess
    glMaterialfv GL_FRONT GL_EMISSION  $no_mat
    glutSolidSphere 1.0 16 16
    glPopMatrix

    # draw sphere in first row, fourth column
    # diffuse reflection; emission; no ambient or specular reflection
    #
    glPushMatrix
    glTranslatef 3.75 3.0 0.0
    glMaterialfv GL_FRONT GL_AMBIENT   $no_mat
    glMaterialfv GL_FRONT GL_DIFFUSE   $mat_diffuse
    glMaterialfv GL_FRONT GL_SPECULAR  $no_mat
    glMaterialfv GL_FRONT GL_SHININESS $no_shininess
    glMaterialfv GL_FRONT GL_EMISSION  $mat_emission
    glutSolidSphere 1.0 16 16
    glPopMatrix

    # draw sphere in second row, first column
    # ambient and diffuse reflection; no specular  
    #
    glPushMatrix
    glTranslatef -3.75 0.0 0.0
    glMaterialfv GL_FRONT GL_AMBIENT   $mat_ambient
    glMaterialfv GL_FRONT GL_DIFFUSE   $mat_diffuse
    glMaterialfv GL_FRONT GL_SPECULAR  $no_mat
    glMaterialfv GL_FRONT GL_SHININESS $no_shininess
    glMaterialfv GL_FRONT GL_EMISSION  $no_mat
    glutSolidSphere 1.0 16 16
    glPopMatrix
 
    # draw sphere in second row, second column
    # ambient, diffuse and specular reflection; low shininess
    #
    glPushMatrix
    glTranslatef -1.25 0.0 0.0
    glMaterialfv GL_FRONT GL_AMBIENT   $mat_ambient
    glMaterialfv GL_FRONT GL_DIFFUSE   $mat_diffuse
    glMaterialfv GL_FRONT GL_SPECULAR  $mat_specular
    glMaterialfv GL_FRONT GL_SHININESS $low_shininess
    glMaterialfv GL_FRONT GL_EMISSION  $no_mat
    glutSolidSphere 1.0 16 16
    glPopMatrix

    # draw sphere in second row, third column
    # ambient, diffuse and specular reflection; high shininess
    #
    glPushMatrix
    glTranslatef 1.25 0.0 0.0
    glMaterialfv GL_FRONT GL_AMBIENT   $mat_ambient
    glMaterialfv GL_FRONT GL_DIFFUSE   $mat_diffuse
    glMaterialfv GL_FRONT GL_SPECULAR  $mat_specular
    glMaterialfv GL_FRONT GL_SHININESS $high_shininess
    glMaterialfv GL_FRONT GL_EMISSION  $no_mat
    glutSolidSphere 1.0 16 16
    glPopMatrix

    # draw sphere in second row, fourth column
    # ambient and diffuse reflection; emission; no specular
    #
    glPushMatrix
    glTranslatef 3.75 0.0 0.0
    glMaterialfv GL_FRONT GL_AMBIENT   $mat_ambient
    glMaterialfv GL_FRONT GL_DIFFUSE   $mat_diffuse
    glMaterialfv GL_FRONT GL_SPECULAR  $no_mat
    glMaterialfv GL_FRONT GL_SHININESS $no_shininess
    glMaterialfv GL_FRONT GL_EMISSION  $mat_emission
    glutSolidSphere 1.0 16 16
    glPopMatrix

    # draw sphere in third row, first column
    # colored ambient and diffuse reflection; no specular  
    #
    glPushMatrix
    glTranslatef -3.75 -3.0 0.0
    glMaterialfv GL_FRONT GL_AMBIENT   $mat_ambient_color
    glMaterialfv GL_FRONT GL_DIFFUSE   $mat_diffuse
    glMaterialfv GL_FRONT GL_SPECULAR  $no_mat
    glMaterialfv GL_FRONT GL_SHININESS $no_shininess
    glMaterialfv GL_FRONT GL_EMISSION  $no_mat
    glutSolidSphere 1.0 16 16
    glPopMatrix

    # draw sphere in third row, second column
    # colored ambient, diffuse and specular reflection; low shininess
    #
    glPushMatrix
    glTranslatef -1.25 -3.0 0.0
    glMaterialfv GL_FRONT GL_AMBIENT   $mat_ambient_color
    glMaterialfv GL_FRONT GL_DIFFUSE   $mat_diffuse
    glMaterialfv GL_FRONT GL_SPECULAR  $mat_specular
    glMaterialfv GL_FRONT GL_SHININESS $low_shininess
    glMaterialfv GL_FRONT GL_EMISSION  $no_mat
    glutSolidSphere 1.0 16 16
    glPopMatrix
 
    # draw sphere in third row, third column
    # colored ambient, diffuse and specular reflection; high shininess
    #
    glPushMatrix
    glTranslatef 1.25 -3.0 0.0
    glMaterialfv GL_FRONT GL_AMBIENT   $mat_ambient_color
    glMaterialfv GL_FRONT GL_DIFFUSE   $mat_diffuse
    glMaterialfv GL_FRONT GL_SPECULAR  $mat_specular
    glMaterialfv GL_FRONT GL_SHININESS $high_shininess
    glMaterialfv GL_FRONT GL_EMISSION  $no_mat
    glutSolidSphere 1.0 16 16
    glPopMatrix

    # draw sphere in third row, fourth column
    # colored ambient and diffuse reflection; emission; no specular
    #
    glPushMatrix
    glTranslatef 3.75 -3.0 0.0
    glMaterialfv GL_FRONT GL_AMBIENT   $mat_ambient_color
    glMaterialfv GL_FRONT GL_DIFFUSE   $mat_diffuse
    glMaterialfv GL_FRONT GL_SPECULAR  $no_mat
    glMaterialfv GL_FRONT GL_SHININESS $no_shininess
    glMaterialfv GL_FRONT GL_EMISSION  $mat_emission
    glutSolidSphere 1.0 16 16
    glPopMatrix
 
    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    if { $w <= [expr $h * 2] } {
       glOrtho -6.0 6.0 [expr -3.0*double($h)*2/double($w)] \
               [expr 3.0*double($h)*2/double($w)] -10.0 10.0
    } else {
       glOrtho [expr -6.0*double($w)/(double($h)*2)] \
               [expr  6.0*double($w)/(double($h)*2)] -3.0 3.0 -10.0 10.0
    }
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 600 -height 450 -double true -depth true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 1
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example material"

bind . <Key-Escape> "exit"

.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
