# list.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program demonstrates how to make and execute a 
# display list.  Note that attributes, such as current 
# color and matrix, are changed.

package require tcl3d

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc CreateCallback { toglwin } {
    global listName

    set listName [glGenLists 1]
    glNewList $listName GL_COMPILE
        glColor3f 1.0 0.0 0.0 ; # current color red
        glBegin GL_TRIANGLES
        glVertex2f 0.0 0.0
        glVertex2f 1.0 0.0
        glVertex2f 0.0 1.0
        glEnd
        glTranslatef 1.5 0.0 0.0 ; # move position
    glEndList
    glShadeModel GL_FLAT
}

proc drawLine {} {
    glBegin GL_LINES
        glVertex2f 0.0 0.5
        glVertex2f 15.0 0.5
    glEnd
}

proc DisplayCallback { toglwin } {
    global listName

    glClear GL_COLOR_BUFFER_BIT

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glColor3f 0.0 1.0 0.0 ; # current color green
    for { set i 0 } { $i < 10 } { incr i } {
        glCallList $listName
    }
    drawLine ; # is this line green?  NO!
    # where is the line drawn?
    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    if { $w <= $h } {
        gluOrtho2D 0.0 2.0 [expr -0.5 * double($h)/double($w)] \
                           [expr 1.5 * double($h)/double($w)]
    } else {
        gluOrtho2D 0.0 [expr 2.0 * double($w)/double($h)] -0.5 1.5 
    }
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 650 -height 50 -double true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 1
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example list"

bind . <Key-Escape> "exit"

.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
