# lines.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program demonstrates geometric primitives and
# their attributes.

package require tcl3d

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc drawOneLine { x1 y1 x2 y2 } {
    glBegin GL_LINES
    glVertex2f $x1 $y1
    glVertex2f $x2 $y2
    glEnd
}

proc CreateCallback { toglwin } {
    glClearColor 0.0 0.0 0.0 0.0
    glShadeModel GL_FLAT
}

proc DisplayCallback { toglwin } {
    glClear GL_COLOR_BUFFER_BIT

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    # select white for all lines
    glColor3f 1.0 1.0 1.0

    # in 1st row, 3 lines, each with a different stipple
    glEnable GL_LINE_STIPPLE
   
    glLineStipple 1 0x0101   ; # dotted
    drawOneLine 50.0 125.0 150.0 125.0
    glLineStipple 1 0x00FF   ; # dashed
    drawOneLine 150.0 125.0 250.0 125.0
    glLineStipple 1 0x1C47   ; # dash/dot/dash
    drawOneLine 250.0 125.0 350.0 125.0

    # in 2nd row, 3 wide lines, each with different stipple
    glLineWidth 5.0
    glLineStipple 1 0x0101    ; # dotted
    drawOneLine 50.0 100.0 150.0 100.0
    glLineStipple 1 0x00FF    ; # dashed
    drawOneLine 150.0 100.0 250.0 100.0
    glLineStipple 1 0x1C47    ; # dash/dot/dash
    drawOneLine 250.0 100.0 350.0 100.0
    glLineWidth 1.0

    # in 3rd row, 6 lines, with dash/dot/dash stipple
    # as part of a single connected line strip
    glLineStipple 1 0x1C47    ; # dash/dot/dash
    glBegin GL_LINE_STRIP
    for { set i 0 } { $i < 7 } { incr i } {
        glVertex2f [expr 50.0 + ($i * 50.0)] 75.0
    }
    glEnd

    #in 4th row, 6 independent lines with same stipple
    for { set i 0 } { $i < 6 } { incr i } {
        drawOneLine [expr 50.0 + ($i * 50.0)] 50.0 \
         [expr 50.0 + (($i+1) * 50.0)] 50.0
    }

    # in 5th row, 1 line, with dash/dot/dash stipple
    # and a stipple repeat factor of 5
    glLineStipple 5 0x1C47   ; # dash/dot/dash
    drawOneLine 50.0 25.0 350.0 25.0

    glDisable GL_LINE_STIPPLE
    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluOrtho2D 0.0 $w 0.0 $h
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 400 -height 150 -double true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 1
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example lines"

bind . <Key-Escape> "exit"

.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
