# image.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program demonstrates drawing pixels and shows the effect
# of glDrawPixels(), glCopyPixels(), and glPixelZoom().
# Interaction: moving the mouse while pressing the mouse button
# will copy the image in the lower-left corner of the window
# to the mouse position, using the current pixel zoom factors.
# There is no attempt to prevent you from drawing over the original
# image.  If you press the 'r' key, the original image and zoom
# factors are reset.  If you press the 'z' or 'Z' keys, you change
# the zoom factors.

package require tcl3d

# Create checkerboard texture
set checkImageWidth 64
set checkImageHeight 64

set checkImage [tcl3dVector GLubyte [expr $checkImageHeight*$checkImageWidth*3]]

set zoomFactor 1.0
set height     0

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# Note: A faster method to calculate and specify textures in Tcl has been 
# introduced with Tcl3D version 0.3. See Tcl3D demo bytearray.tcl.
proc makeCheckImage {} {
    for { set i 0 } { $i < $::checkImageHeight } { incr i } {
        for { set j 0 } { $j < $::checkImageWidth } { incr j } { 
         set c [expr {(((($i&0x8)==0)^(($j&0x8))==0))*255}]
         $::checkImage set [expr {($i*$::checkImageWidth + $j)*3 + 0}] $c
         $::checkImage set [expr {($i*$::checkImageWidth + $j)*3 + 1}] $c
         $::checkImage set [expr {($i*$::checkImageWidth + $j)*3 + 2}] $c
      }
   }
}

proc CreateCallback { toglwin } {    
    glClearColor 0.0 0.0 0.0 0.0
    glShadeModel GL_FLAT

    makeCheckImage
    glPixelStorei GL_UNPACK_ALIGNMENT 1
}

proc DisplayCallback { toglwin } {
    glClear GL_COLOR_BUFFER_BIT
    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glRasterPos2i 0 0
    glDrawPixels $::checkImageWidth $::checkImageHeight GL_RGB \
                 GL_UNSIGNED_BYTE $::checkImage
    glFlush
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    set ::height $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluOrtho2D 0.0 $w 0.0 $h
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
}

proc UpdateMsg { str } {
    .fr.usage configure -state normal
    .fr.usage delete end
    .fr.usage insert end $str
    .fr.usage configure -state disabled
}

proc ResetZoom {} {
    set ::zoomFactor 1.0
    UpdateMsg [format "zoomFactor is now %4.1f" $::zoomFactor]
    .fr.toglwin postredisplay
}

proc IncrZoom {} {
    set ::zoomFactor [expr $::zoomFactor + 0.5]
    if { $::zoomFactor >= 3.0 } {
        set ::zoomFactor 3.0
    }
    UpdateMsg [format "zoomFactor is now %4.1f" $::zoomFactor]
}

proc DecrZoom {} {
    set ::zoomFactor [expr $::zoomFactor - 0.5]
    if { $::zoomFactor <= 0.5 } {
        set ::zoomFactor 0.5
    }
    UpdateMsg [format "zoomFactor is now %4.1f" $::zoomFactor]
}

proc Motion { x y } {
    set ::screeny [expr $::height - $y]
    glRasterPos2i $x $::screeny
    glPixelZoom $::zoomFactor $::zoomFactor
    glCopyPixels 0 0 $::checkImageWidth $::checkImageHeight GL_COLOR
    glPixelZoom 1.0 1.0
    glFlush
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 400 -height 400 -double false \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 6
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example image"

bind . <Key-Escape> "exit"
bind . <Key-r> "ResetZoom"
bind . <Key-z> "IncrZoom"
bind . <Key-Z> "DecrZoom"

bind .fr.toglwin <B1-Motion> "Motion %x %y"

.fr.usage insert end "Key-r      Reset zoom"
.fr.usage insert end "Key-z      Increment zoom"
.fr.usage insert end "Key-Z      Decrement zoom"
.fr.usage insert end "Mouse-L    Paint"
.fr.usage insert end "Key-Escape Exit"
.fr.usage insert end "Initial zoom factor is $::zoomFactor"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
