# font.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# Draws some text in a bitmapped font.  Uses glBitmap() 
# and other pixel routines.  Also demonstrates use of 
# display lists.

package require tcl3d

set space {
    0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 }

set letters {
    {0x00 0x00 0xc3 0xc3 0xc3 0xc3 0xff 0xc3 0xc3 0xc3 0x66 0x3c 0x18} 
    {0x00 0x00 0xfe 0xc7 0xc3 0xc3 0xc7 0xfe 0xc7 0xc3 0xc3 0xc7 0xfe} 
    {0x00 0x00 0x7e 0xe7 0xc0 0xc0 0xc0 0xc0 0xc0 0xc0 0xc0 0xe7 0x7e} 
    {0x00 0x00 0xfc 0xce 0xc7 0xc3 0xc3 0xc3 0xc3 0xc3 0xc7 0xce 0xfc} 
    {0x00 0x00 0xff 0xc0 0xc0 0xc0 0xc0 0xfc 0xc0 0xc0 0xc0 0xc0 0xff} 
    {0x00 0x00 0xc0 0xc0 0xc0 0xc0 0xc0 0xc0 0xfc 0xc0 0xc0 0xc0 0xff} 
    {0x00 0x00 0x7e 0xe7 0xc3 0xc3 0xcf 0xc0 0xc0 0xc0 0xc0 0xe7 0x7e} 
    {0x00 0x00 0xc3 0xc3 0xc3 0xc3 0xc3 0xff 0xc3 0xc3 0xc3 0xc3 0xc3} 
    {0x00 0x00 0x7e 0x18 0x18 0x18 0x18 0x18 0x18 0x18 0x18 0x18 0x7e} 
    {0x00 0x00 0x7c 0xee 0xc6 0x06 0x06 0x06 0x06 0x06 0x06 0x06 0x06} 
    {0x00 0x00 0xc3 0xc6 0xcc 0xd8 0xf0 0xe0 0xf0 0xd8 0xcc 0xc6 0xc3} 
    {0x00 0x00 0xff 0xc0 0xc0 0xc0 0xc0 0xc0 0xc0 0xc0 0xc0 0xc0 0xc0} 
    {0x00 0x00 0xc3 0xc3 0xc3 0xc3 0xc3 0xc3 0xdb 0xff 0xff 0xe7 0xc3} 
    {0x00 0x00 0xc7 0xc7 0xcf 0xcf 0xdf 0xdb 0xfb 0xf3 0xf3 0xe3 0xe3} 
    {0x00 0x00 0x7e 0xe7 0xc3 0xc3 0xc3 0xc3 0xc3 0xc3 0xc3 0xe7 0x7e} 
    {0x00 0x00 0xc0 0xc0 0xc0 0xc0 0xc0 0xfe 0xc7 0xc3 0xc3 0xc7 0xfe} 
    {0x00 0x00 0x3f 0x6e 0xdf 0xdb 0xc3 0xc3 0xc3 0xc3 0xc3 0x66 0x3c} 
    {0x00 0x00 0xc3 0xc6 0xcc 0xd8 0xf0 0xfe 0xc7 0xc3 0xc3 0xc7 0xfe} 
    {0x00 0x00 0x7e 0xe7 0x03 0x03 0x07 0x7e 0xe0 0xc0 0xc0 0xe7 0x7e} 
    {0x00 0x00 0x18 0x18 0x18 0x18 0x18 0x18 0x18 0x18 0x18 0x18 0xff} 
    {0x00 0x00 0x7e 0xe7 0xc3 0xc3 0xc3 0xc3 0xc3 0xc3 0xc3 0xc3 0xc3} 
    {0x00 0x00 0x18 0x3c 0x3c 0x66 0x66 0xc3 0xc3 0xc3 0xc3 0xc3 0xc3} 
    {0x00 0x00 0xc3 0xe7 0xff 0xff 0xdb 0xdb 0xc3 0xc3 0xc3 0xc3 0xc3} 
    {0x00 0x00 0xc3 0x66 0x66 0x3c 0x3c 0x18 0x3c 0x3c 0x66 0x66 0xc3} 
    {0x00 0x00 0x18 0x18 0x18 0x18 0x18 0x18 0x3c 0x3c 0x66 0x66 0xc3} 
    {0x00 0x00 0xff 0xc0 0xc0 0x60 0x30 0x7e 0x0c 0x06 0x03 0x03 0xff}
}

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc makeRasterFont {} {
    global fontOffset

    glPixelStorei GL_UNPACK_ALIGNMENT 1

    set fontOffset [glGenLists 128]
    set j [tcl3dCharToNum "A"]
    set len [llength $::letters]
    for { set i 0 } { $i < $len } { incr i } {
        glNewList [expr $fontOffset + $j] GL_COMPILE
            glBitmap 8 13 0.0 2.0 10.0 0.0 [lindex $::letters $i]
        glEndList
        incr j
    }
    glNewList [expr $fontOffset + [tcl3dCharToNum " "]] GL_COMPILE
    glBitmap 8 13 0.0 2.0 10.0 0.0 $::space
    glEndList
}

proc CreateCallback { toglwin } {
    glShadeModel GL_FLAT
    makeRasterFont
}

proc printString { s } {
    global fontOffset

    set len [string length $s]
    if { $len > 0 } {
        glPushAttrib GL_LIST_BIT
        glListBase $fontOffset
        set sa [tcl3dVectorFromString GLubyte $s]
        glCallLists $len GL_UNSIGNED_BYTE $sa
        $sa delete
        glPopAttrib
    }
}

# Everything above this line could be in a library 
# that defines a font.  To make it work, you've got 
# to call makeRasterFont() before you start making 
# calls to printString().
#
proc DisplayCallback { toglwin } {
    set white { 1.0 1.0 1.0 }

    glClear GL_COLOR_BUFFER_BIT

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glColor3fv $white

    glRasterPos2i 20 60
    printString "THE QUICK BROWN FOX JUMPS"
    glRasterPos2i 20 40
    printString "OVER A LAZY DOG"
    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    glOrtho 0.0 $w 0.0 $h -1.0 1.0
    glMatrixMode GL_MODELVIEW
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 300 -height 100 -double true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 1
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example font"

bind . <Key-Escape> "exit"

.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
