# double.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This is a simple double buffered program.
# Pressing the left mouse button rotates the rectangle.
# Pressing the middle mouse button stops the rotation.

package require tcl3d

set spin 0.0

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# Put all exit related code here.
proc ExitProg {} {
    exit
}

proc DisplayCallback { toglwin } {
    glClear GL_COLOR_BUFFER_BIT

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glPushMatrix
    glRotatef $::spin 0.0 0.0 1.0
    glColor3f 1.0 1.0 1.0
    glRectf -25.0 -25.0 25.0 25.0
    glPopMatrix
    glFlush
 
    $toglwin swapbuffers
}

proc StartAnimation {} {
    global spin

    set spin [expr $spin + 2.0]
    if { $spin > 360.0 } {
        set spin [expr $spin - 360.0]
    }
    .fr.toglwin postredisplay
    set ::spinId [tcl3dAfterIdle StartAnimation]
}

proc StopAnimation {} {
    if { [info exists ::spinId] } {
        after cancel $::spinId 
    }
}

proc CreateCallback { toglwin } {
    glClearColor 0.0 0.0 0.0 0.0
    glShadeModel GL_FLAT
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    glOrtho -50.0 50.0 -50.0 50.0 -1.0 1.0
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 500 -height 500 -double true \
                 -swapinterval 1 \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 3
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example double"

bind . <Key-Escape> "exit"
bind .fr.toglwin <1> "StartAnimation"
bind .fr.toglwin <2> "StopAnimation" 
bind .fr.toglwin <3> "StopAnimation" 
bind .fr.toglwin <Control-Button-1> "StopAnimation"

.fr.usage insert end "Mouse-L    Increase animation speed"
.fr.usage insert end "Mouse-MR   Decrease animation speed"
.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
if { [file tail [info script]] eq [file tail $::argv0] } {
    # If started directly from tclsh or wish, then start animation.
    update
    StartAnimation
}
