# colormat.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# After initialization, the program will be in
# ColorMaterial mode. Interaction:  pressing the 
# mouse buttons will change the diffuse reflection values.

package require tcl3d

set diffuseMaterial { 0.5 0.5 0.5 1.0 }

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc AddEvents {} {
    event add <<LeftMousePress>> <ButtonPress-1>
    if { $::tcl_platform(os) eq "Darwin" } {
        event add <<MiddleMousePress>> <ButtonPress-3>
        event add <<RightMousePress>>  <ButtonPress-2>
        event add <<RightMousePress>>  <Control-ButtonPress-1>
    } else {
        event add <<MiddleMousePress>> <ButtonPress-2>
        event add <<RightMousePress>>  <ButtonPress-3>
    }
}

# Initialize material property, light source, lighting model,
# and depth buffer.
#
proc CreateCallback { toglwin } {
    global diffuseMaterial

    set mat_specular   { 1.0 1.0 1.0 1.0 }
    set light_position { 1.0 1.0 1.0 0.0 }

    glClearColor 0.0 0.0 0.0 0.0
    glShadeModel GL_SMOOTH
    glEnable GL_DEPTH_TEST
    glMaterialfv GL_FRONT GL_DIFFUSE $diffuseMaterial
    glMaterialfv GL_FRONT GL_SPECULAR $mat_specular
    glMaterialf GL_FRONT GL_SHININESS 25.0
    glLightfv GL_LIGHT0 GL_POSITION $light_position
    glEnable GL_LIGHTING
    glEnable GL_LIGHT0

    glColorMaterial GL_FRONT GL_DIFFUSE
    glEnable GL_COLOR_MATERIAL
}

proc DisplayCallback { toglwin } {
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glutSolidSphere 1.0 20 16
    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    if { $w <= $h } {
        glOrtho -1.5 1.5 [expr -1.5*double($h)/double($w)] \
                [expr 1.5*double($h)/double($w)] -10.0 10.0
    } else {
        glOrtho [expr -1.5*double($w)/double($h)] \
                [expr 1.5*double($w)/double($h)] -1.5 1.5 -10.0 10.0
    }
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
}

proc changeDiffuse { comp } {
    global diffuseMaterial

    lset diffuseMaterial $comp [expr [lindex $diffuseMaterial $comp] + 0.1]
    if { [lindex $diffuseMaterial $comp] > 1.0 } {
       lset diffuseMaterial $comp 0.0
    }
    glColor4fv $diffuseMaterial
    .fr.toglwin postredisplay
}

proc ChangeRedDiffuse {} {
    changeDiffuse 0
}

proc ChangeGreenDiffuse {} {
    changeDiffuse 1
}

proc ChangeBlueDiffuse {} {
    changeDiffuse 2
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 500 -height 500 -double true -depth true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 4
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example colormat"

AddEvents

bind .fr.toglwin <<LeftMousePress>>   "ChangeRedDiffuse"
bind .fr.toglwin <<MiddleMousePress>> "ChangeGreenDiffuse"
bind .fr.toglwin <<RightMousePress>>  "ChangeBlueDiffuse"

bind . <Key-Escape> "exit"

.fr.usage insert end "Mouse-L    Change red   diffuse"
.fr.usage insert end "Mouse-M    Change green diffuse"
.fr.usage insert end "Mouse-R    Change blue  diffuse"
.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
