# blendeqn.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# Demonstrate the different blending functions available with the
# OpenGL imaging subset.  This program demonstrates use of the
# glBlendEquation call.
#
# The following keys change the selected blend equation function:
#
#      'a'  ->  GL_FUNC_ADD
#      's'  ->  GL_FUNC_SUBTRACT
#      'r'  ->  GL_FUNC_REVERSE_SUBTRACT
#      'm'  ->  GL_MIN
#      'x'  ->  GL_MAX

package require tcl3d

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc CreateCallback { toglwin } {
   glClearColor 1.0 1.0 0.0 0.0

   glBlendFunc GL_ONE GL_ONE
   glEnable GL_BLEND
}

proc DisplayCallback { toglwin } {
    glClear GL_COLOR_BUFFER_BIT

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glColor3f 0.0 0.0 1.0
    glRectf -0.5 -0.5 0.5 0.5

    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    set aspect [expr double ($w) / double ($h)]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    if {$aspect < 1.0} {
        set aspect [expr 1.0 / $aspect]
        glOrtho [expr -1.0 * $aspect] $aspect -1.0 1.0 -1.0 1.0
    } else {
        glOrtho -1.0 1.0 [expr -1.0 * $aspect] $aspect -1.0 1.0
    }
    glMatrixMode GL_MODELVIEW
}

proc UpdateMsg { msgStr } {
    .fr.usage configure -state normal
    .fr.usage delete end
    .fr.usage insert end $msgStr
    .fr.usage configure -state disabled
}

proc setBlendEquation { equat msgStr } {
    $::glBlendEquationFunc $equat
    UpdateMsg $msgStr
    .fr.toglwin postredisplay
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 512 -height 512 -double true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 7
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example blendeqn"

set msg {"white square on yellow background"}
bind . <Key-a> "setBlendEquation $::GL_FUNC_ADD $msg"

set msg {"blue square on yellow background"}
bind . <Key-s> "setBlendEquation $::GL_FUNC_SUBTRACT $msg"

set msg {"yellow square on yellow background"}
bind . <Key-r> "setBlendEquation $::GL_FUNC_REVERSE_SUBTRACT $msg"

set msg {"black square on yellow background"}
bind . <Key-m> "setBlendEquation $::GL_MIN $msg"

set msg {"white square on yellow background"}
bind . <Key-x> "setBlendEquation $::GL_MAX $msg"

bind . <Key-Escape> "exit"

.fr.usage insert end "Key-a      GL_FUNC_ADD"
.fr.usage insert end "Key-s      GL_FUNC_SUBTRACT"
.fr.usage insert end "Key-r      GL_FUNC_REVERSE_SUBTRACT"
.fr.usage insert end "Key-m      GL_MIN"
.fr.usage insert end "Key-x      GL_MAX"
.fr.usage insert end "Key-Escape Exit"
.fr.usage insert end "Default is GL_FUNC_ADD"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]

if { ! [tcl3dOglHaveVersion 1 2] } {
    tk_messageBox -icon warning -type ok -title "Invalid OpenGL version" \
                  -message [format "Feature needs OpenGL >= 1.2. Only have %s" \
                            [glGetString GL_VERSION]]
}

set glBlendEquationFunc [tcl3dOglFindFunc "glBlendEquation"]
if { $glBlendEquationFunc eq "" } {
    tk_messageBox -icon error -type ok -title "Missing OpenGL function" \
                  -message "OpenGL function glBlendEquation missing"
    exit
}
