# bezmesh.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program renders a lighted, filled Bezier surface,
# using two-dimensional evaluators.

package require tcl3d

set ctrlpoints {
   { {-1.5 -1.5 4.0}
     {-0.5 -1.5 2.0}
     {0.5 -1.5 -1.0}
     {1.5 -1.5 2.0}}
   { {-1.5 -0.5 1.0}
     {-0.5 -0.5 3.0}
     {0.5 -0.5 0.0}
     {1.5 -0.5 -1.0}}
   { {-1.5 0.5 4.0}
     {-0.5 0.5 0.0}
     {0.5 0.5 3.0}
     {1.5 0.5 4.0}}
   { {-1.5 1.5 -2.0}
     {-0.5 1.5 -2.0}
     {0.5 1.5 0.0}
     {1.5 1.5 -1.0}}
}

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc initlights {} {
   set ambient {0.2 0.2 0.2 1.0}
   set position {0.0 0.0 2.0 1.0}
   set mat_diffuse {0.6 0.6 0.6 1.0}
   set mat_specular {1.0 1.0 1.0 1.0}
   set mat_shininess {50.0}

   glEnable GL_LIGHTING
   glEnable GL_LIGHT0

   glLightfv GL_LIGHT0 GL_AMBIENT  $ambient
   glLightfv GL_LIGHT0 GL_POSITION $position

   glMaterialfv GL_FRONT GL_DIFFUSE   $mat_diffuse
   glMaterialfv GL_FRONT GL_SPECULAR  $mat_specular
   glMaterialfv GL_FRONT GL_SHININESS $mat_shininess
}

proc DisplayCallback { toglwin } {
   glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

   glPushMatrix
   glRotatef 85.0 1.0 1.0 1.0
   glEvalMesh2 GL_FILL 0 20 0 20
   glPopMatrix
   glFlush
   $toglwin swapbuffers
}

proc CreateCallback { toglwin } {
   glClearColor 0.0 0.0 0.0 0.0
   glEnable GL_DEPTH_TEST
   glMap2f GL_MAP2_VERTEX_3 0 1 3 4 \
           0 1 12 4 [join [join $::ctrlpoints]]
   glEnable GL_MAP2_VERTEX_3
   glEnable GL_AUTO_NORMAL
   glMapGrid2f 20 0.0 1.0 20 0.0 1.0
   initlights
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    if { $w <= $h } {
       glOrtho -4.0 4.0 \
                [expr -4.0*double($h)/double($w)] \
                [expr  4.0*double($h)/double($w)] \
                -4.0 4.0
    } else {
        glOrtho [expr -4.0*double($w)/double($h)] \
                [expr  4.0*double($w)/double($h)] \
                -4.0 4.0 -4.0 4.0
    }
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 500 -height 500 -double true -depth true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 1
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example bezmesh"

bind . <Key-Escape> "exit"

.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
