# alpha.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program draws several overlapping filled polygons
# to demonstrate the effect order has on alpha blending results.
# Use the 't' key to toggle the order of drawing polygons.

package require tcl3d

set leftFirst 1

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc CreateCallback { toglwin } {
   glEnable GL_BLEND
   glBlendFunc GL_SRC_ALPHA GL_ONE_MINUS_SRC_ALPHA
   glShadeModel GL_FLAT
   glClearColor 0.0 0.0 0.0 0.0
}

proc drawLeftTriangle {} {
   # draw yellow triangle on LHS of screen

   glBegin GL_TRIANGLES
      glColor4f 1.0 1.0 0.0 0.75
      glVertex3f 0.1 0.9 0.0
      glVertex3f 0.1 0.1 0.0 
      glVertex3f 0.7 0.5 0.0 
   glEnd
}

proc drawRightTriangle {} {
   # draw cyan triangle on RHS of screen

   glBegin GL_TRIANGLES
      glColor4f 0.0 1.0 1.0 0.75
      glVertex3f 0.9 0.9 0.0 
      glVertex3f 0.3 0.5 0.0 
      glVertex3f 0.9 0.1 0.0 
   glEnd
}

proc DisplayCallback { toglwin } {
   glClear GL_COLOR_BUFFER_BIT

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

   if { $::leftFirst } {
      drawLeftTriangle
      drawRightTriangle
   } else {
      drawRightTriangle
      drawLeftTriangle
   }

   glFlush
   $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    if { $w <= $h } {
        gluOrtho2D 0.0 1.0 0.0 [expr 1.0*double($h)/double($w)]
    } else {
        gluOrtho2D 0.0 [expr 1.0*double($w)/double($h)] 0.0 1.0
    }
}

proc ToggleWinOrder {} {
    set ::leftFirst [expr ! $::leftFirst]
    .fr.toglwin postredisplay
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 400 -height 400 -double true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 2
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example alpha"

bind . <Key-t>      "ToggleWinOrder"
bind . <Key-Escape> "exit"

.fr.usage insert end "Key-t      Toggle polygon order"
.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
