# aargb.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program draws shows how to draw anti-aliased lines. It draws
# two diagonal lines to form an X; when 'r' is typed in the window, 
# the lines are rotated in opposite directions.

package require tcl3d

set rotAngle 0.0

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# Initialize antialiasing for RGBA mode, including alpha
# blending, hint, and line width.  Print out implementation
# specific info on line width granularity and width.
#
proc CreateCallback { toglwin } {
    global lw_granularity lw_range

    set lw_granularity [tcl3dOglGetFloatState GL_LINE_WIDTH_GRANULARITY 2]
    set lw_range       [tcl3dOglGetFloatState GL_LINE_WIDTH_RANGE 2]

    glEnable GL_LINE_SMOOTH
    glEnable GL_BLEND
    glBlendFunc GL_SRC_ALPHA GL_ONE_MINUS_SRC_ALPHA
    glHint GL_LINE_SMOOTH_HINT GL_DONT_CARE
    glLineWidth 1.5

    glClearColor 0.0 0.0 0.0 0.0
}

# Draw 2 diagonal lines to form an X
#
proc DisplayCallback { toglwin } {
    global rotAngle

    glClear GL_COLOR_BUFFER_BIT

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glColor3f 0.0 1.0 0.0
    glPushMatrix
    glRotatef [expr -1.0 * $rotAngle] 0.0 0.0 0.1
    glBegin GL_LINES
        glVertex2f -0.5 0.5
        glVertex2f 0.5 -0.5
    glEnd
    glPopMatrix

    glColor3f 0.0 0.0 1.0
    glPushMatrix
    glRotatef $rotAngle 0.0 0.0 0.1
    glBegin GL_LINES
        glVertex2f 0.5 0.5
        glVertex2f -0.5 -0.5
    glEnd
    glPopMatrix

    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    if { $w <= $h } {
        gluOrtho2D -1.0 1.0 [expr -1.0*$h/$w] [expr 1.0*$h/$w]
    } else {
        gluOrtho2D [expr -1.0*$w/$h] [expr 1.0*$w/$h] -1.0 1.0
    }
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
}

proc IncrRotAngle {} {
    set ::rotAngle [expr $::rotAngle + 20.0]
    if { $::rotAngle >= 360.0 } {
        set ::rotAngle 0.0
    }
    .fr.toglwin postredisplay
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 200 -height 200 -double true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 4
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example aargb"

bind . <Key-r> "IncrRotAngle"
bind . <Key-Escape> "exit"

.fr.usage insert end "Key-r      Rotate"
.fr.usage insert end "Key-Escape Exit"
.fr.usage insert end "GL_LINE_WIDTH_GRANULARITY: [lindex $lw_granularity 0]"
.fr.usage insert end "GL_LINE_WIDTH_RANGE: [lindex $lw_range 0] [lindex $lw_range 1]"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
