# aapoly.tcl
#
# An example of the OpenGL red book modified to work with Tcl3D.
# The original C sources are Copyright (c) 1993-2003, Silicon Graphics, Inc.
# The Tcl3D sources are Copyright (c) 2005-2025, Paul Obermeier.
# See file LICENSE for complete license information.
#
# This program draws filled polygons with antialiased
# edges.  The special GL_SRC_ALPHA_SATURATE blending 
# function is used.
# Pressing the 't' key turns the antialiasing on and off.

package require tcl3d

set polySmooth 1

set NFACE 6

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc CreateCallback { toglwin } {
    glCullFace GL_BACK
    glEnable GL_CULL_FACE
    glBlendFunc GL_SRC_ALPHA_SATURATE GL_ONE
    glClearColor 0.0 0.0 0.0 0.0
}

proc drawCube { x0 x1 y0 y1 z0 z1 } {
    # indices of front, top, left, bottom, right, back faces
    set indices {
        {4 5 6 7} {2 3 7 6} {0 4 7 3}
        {0 1 5 4} {1 5 6 2} {0 3 2 1}
    }

    set v [list \
        [list $x0 $y0 $z0 0.0 0.0 0.0 1.0] \
        [list $x1 $y0 $z0 1.0 0.0 0.0 1.0] \
        [list $x1 $y1 $z0 0.0 1.0 0.0 1.0] \
        [list $x0 $y1 $z0 1.0 1.0 0.0 1.0] \
        [list $x0 $y0 $z1 0.0 0.0 1.0 1.0] \
        [list $x1 $y0 $z1 1.0 0.0 1.0 1.0] \
        [list $x1 $y1 $z1 0.0 1.0 1.0 1.0] \
        [list $x0 $y1 $z1 1.0 1.0 1.0 1.0] \
    ]

    set va [tcl3dVectorFromList GLfloat [join [join $v]]]
    set ca [tcl3dVectorInd $va GLfloat 3]
    set ia [tcl3dVectorFromList GLubyte [join [join $indices]]]

    glEnableClientState GL_VERTEX_ARRAY
    glEnableClientState GL_COLOR_ARRAY
    glVertexPointer 3 GL_FLOAT [expr 4*7] $va
    glColorPointer  4 GL_FLOAT [expr 4*7] $ca]
    glDrawElements GL_QUADS [expr $::NFACE*4] GL_UNSIGNED_BYTE $ia
    glDisableClientState GL_VERTEX_ARRAY
    glDisableClientState GL_COLOR_ARRAY
}

# Note:  polygons must be drawn from front to back for proper blending.

proc DisplayCallback { toglwin } {
    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    if { $::polySmooth } {
        glClear   GL_COLOR_BUFFER_BIT
        glEnable  GL_BLEND
        glEnable  GL_POLYGON_SMOOTH
        glDisable GL_DEPTH_TEST
    } else { 
        glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]
        glDisable GL_BLEND
        glDisable GL_POLYGON_SMOOTH
        glEnable  GL_DEPTH_TEST
    }

    glPushMatrix
        glTranslatef 0.0 0.0 -8.0
        glRotatef 30.0 1.0 0.0 0.0
        glRotatef 60.0 0.0 1.0 0.0
        drawCube -0.5 0.5 -0.5 0.5 -0.5 0.5
    glPopMatrix

    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluPerspective 30.0 [expr double($w)/double($h)] 1.0 20.0
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
}

proc ToggleSmooth {} {
    set ::polySmooth [expr ! $::polySmooth]
    .fr.toglwin postredisplay
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 400 -height 400 \
                 -double true -depth true -alpha true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::listFont -height 2
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: OpenGL Red Book example aapolyStride"

bind . <Key-Escape> "exit"
bind . <Key-t> "ToggleSmooth"

.fr.usage insert end "Key-t      Toggle smoothing"
.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
