#
# OpenGL 3.3 with GLEW - Example 04
#
# @author   Norbert Nopper norbert@nopper.tv
# @version  1.0
#
# Homepage: https://github.com/McNopper/OpenGL
#
# Copyright Norbert Nopper
#
# Modified for Tcl3D by Paul Obermeier 2010/09/01
# See www.tcl3d.org for the Tcl3D extension.

package require Tk
package require tcl3d

# Font to be used in the Tk listbox.
set g_Demo(listFont) {-family {Courier} -size 10}

# Obtain the name of this script file.
set g_Demo(scriptDir) [file dirname [info script]]

# Window size.
set g_Demo(winWidth)  640
set g_Demo(winHeight) 480

# Projection matrix. 
set g_Demo(projection) [tcl3dVector GLfloat 16]

# ModelView matrix.
set g_Demo(modelView) [tcl3dVector GLfloat 16]

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    ExitProg
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# Function for initialization.
proc Init {} {
    global g_Demo g_Program

    # Load the source of the vertex shader.
    set vertexSource [tcl3dOglReadShaderFile [file join $g_Demo(scriptDir) "Vertex.vs"]]

    # Load the source of the fragment shader.
    set fragmentSource [tcl3dOglReadShaderFile [file join $g_Demo(scriptDir) "Fragment.fs"]]

    set g_Program [tcl3dOglBuildProgram $vertexSource "" "" "" $fragmentSource]
    set program [dict get $g_Program program]

    # The VAO for the vertices etc.
    set g_Demo(vao) [tcl3dVector GLuint 1]
    glGenVertexArrays 1 $g_Demo(vao)
    glBindVertexArray [$g_Demo(vao) get 0]

    set sphere [glusCreateSphere 1.0 32]
    set g_Demo(numIndices) [dict get $sphere numIndices]

    # Location of the projection matrix in the shader program.
    set g_Demo(projectionLocation) [glGetUniformLocation $program "projectionMatrix"]

    # Location of the model view matrix in the shader program.
    set g_Demo(modelViewLocation) [glGetUniformLocation $program "modelViewMatrix"]

    # The location of the vertex in the shader program.
    set vertexLocation [glGetAttribLocation $program "vertex"]

    # The location of the normal in the shader program.
    set normalLocation [glGetAttribLocation $program "normal"]

    # The VBO for the vertices.
    set g_Demo(vertices) [tcl3dVector GLuint 1]
    glGenBuffers 1 $g_Demo(vertices)
    glBindBuffer GL_ARRAY_BUFFER [$g_Demo(vertices) get 0]
    set vertexVec [dict get $sphere vertexVec]
    glBufferData GL_ARRAY_BUFFER \
                 [expr [dict get $sphere numVertices]*4*[$vertexVec elemsize]] \
                 $vertexVec GL_STATIC_DRAW

    set g_Demo(normals) [tcl3dVector GLuint 1]
    glGenBuffers 1 $g_Demo(normals)
    glBindBuffer GL_ARRAY_BUFFER [$g_Demo(normals) get 0]
    set normalVec [dict get $sphere normalVec]
    glBufferData GL_ARRAY_BUFFER \
                 [expr [dict get $sphere numVertices]*3*[$normalVec elemsize]] \
                 $normalVec GL_STATIC_DRAW

    set g_Demo(indices) [tcl3dVector GLuint 1]
    glGenBuffers 1 $g_Demo(indices)
    glBindBuffer GL_ELEMENT_ARRAY_BUFFER [$g_Demo(indices) get 0]
    set indexVec [dict get $sphere indexVec]
    glBufferData GL_ELEMENT_ARRAY_BUFFER \
                 [expr [dict get $sphere numIndices]*[$indexVec elemsize]] \
                 $indexVec GL_STATIC_DRAW

    glusDestroyShape $sphere

    glUseProgram $program

    # Matrix for the model
    set model [tcl3dVector GLfloat 16]
    set rot1  [tcl3dVector GLfloat 16]
    set rot2  [tcl3dVector GLfloat 16]

    # Calculate the model matrix ...
    tcl3dMatfIdentity $model

    tcl3dMatfRotateY 30 $rot1
    tcl3dMatfRotateX 30 $rot2
    tcl3dMatfMult $rot1 $rot2 $model

    # ... and the view matrix ...
    tcl3dLookAt 0.0 0.0 5.0  0.0 0.0 0.0  0.0 1.0 0.0  $g_Demo(modelView)

    # ... to get the final model view matrix
    tcl3dMatfMult $g_Demo(modelView) $model $g_Demo(modelView)

    $model delete
    $rot1  delete
    $rot2  delete

    set modelViewAsList [tcl3dVectorToList $g_Demo(modelView) 16]
    glUniformMatrix4fv $g_Demo(modelViewLocation) 1 GL_FALSE $modelViewAsList

    glBindBuffer GL_ARRAY_BUFFER [$g_Demo(vertices) get 0]
    glVertexAttribPointer $vertexLocation 4 GL_FLOAT GL_FALSE 0 "NULL"
    glEnableVertexAttribArray $vertexLocation

    glBindBuffer GL_ARRAY_BUFFER [$g_Demo(normals) get 0]
    glVertexAttribPointer $normalLocation 3 GL_FLOAT GL_FALSE 0 "NULL"
    glEnableVertexAttribArray $normalLocation
}

proc CreateCallback { toglwin } {
    glClearColor 0.0 0.0 0.0 0.0
    glClearDepth 1.0
    glEnable GL_DEPTH_TEST
    glEnable GL_CULL_FACE
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    global g_Demo

    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h

    if { ! $g_Demo(haveNeededVersion) } {
        return
    }
    # Calculate the projection matrix and set it
    tcl3dPerspective 40.0 [expr {double($w)/double($h) }] \
                     1.0 100.0 $g_Demo(projection)
    set projectionAsList [tcl3dVectorToList $g_Demo(projection) 16]
    glUniformMatrix4fv $g_Demo(projectionLocation) 1 GL_FALSE $projectionAsList
}

proc DisplayCallback { toglwin } {
    global g_Demo

    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    if { ! $g_Demo(haveNeededVersion) } {
        $toglwin swapbuffer
        return
    }

    glDrawElements GL_TRIANGLES $g_Demo(numIndices) GL_UNSIGNED_INT "NULL"
    $toglwin swapbuffer
}

proc Cleanup {} {
    global g_Demo g_Program

    if { [info exists g_Demo(vertices)] } {
        glDeleteBuffers 1 [$g_Demo(vertices) get 0]
        $g_Demo(vertices) delete
    }

    if { [info exists g_Demo(normals)] } {
        glDeleteBuffers 1 [$g_Demo(normals) get 0]
        $g_Demo(normals) delete
    }

    if { [info exists g_Demo(indices)] } {
        glDeleteBuffers 1 [$g_Demo(indices) get 0]
        $g_Demo(indices) delete
    }

    if { [info exists g_Demo(vao)] } {
        glDeleteVertexArrays 1 [$g_Demo(vao) get 0]
        $g_Demo(vao) delete
    }

    if { [info exists g_Program] } {
        tcl3dOglDestroyProgram $g_Program
    }

    # Unset all global variables. 
    # Needed when running the demo in the Tcl3D presentation framework.
    foreach var [info globals g_*] {
        uplevel #0 unset $var
    }
}

# Put all exit related code here.
proc ExitProg {} {
    exit
}

# Create the OpenGL window and some Tk helper widgets.
proc CreateWindow {} {
    global g_Demo

    frame .fr
    pack .fr -expand 1 -fill both

    # Create a Togl window with an OpenGL core profile using version 3.3.
    # Reshape and Display callbacks are configured later after knowing if
    # the needed OpenGL profile version is available.
    togl .fr.toglwin -width $g_Demo(winWidth) -height $g_Demo(winHeight) \
                     -double true -depth true \
                     -createcommand CreateCallback \
                     -coreprofile true -major 3 -minor 3

    set g_Demo(haveNeededVersion) true
    set numRows 1
    set haveGL3 [tcl3dOglHaveVersion 3]
    if { ! $haveGL3 } {
        set msgStr [format \
            "Demo needs core profile 3.3. Only have GL version %s" \
            [tcl3dOglGetVersion]]
        set g_Demo(haveNeededVersion) false
        incr numRows
    } else {
        set profile [tcl3dOglGetProfile .fr.toglwin]
        if { [dict get $profile "coreprofile"] != true } {
            set msgStr [format \
                "Demo needs core profile 3.3. Only have compatibility profile %d.%d" \
                [dict get $profile "major"] \
                [dict get $profile "minor"]]
            incr numRows
        }
    }
    if { $g_Demo(haveNeededVersion) } {
        # If OpenGL 3.3 or higher is available, initialize the buffers.
        Init
    }

    # Now attach the Reshape and Display callbacks to the Togl window.
    .fr.toglwin configure \
        -reshapecommand ReshapeCallback \
        -displaycommand DisplayCallback 

    listbox .fr.usage -font $g_Demo(listFont) -height $numRows
    label .fr.info
    grid .fr.toglwin -row 0 -column 0 -sticky news
    grid .fr.usage   -row 1 -column 0 -sticky news
    grid .fr.info    -row 2 -column 0 -sticky news
    grid rowconfigure .fr 0 -weight 1
    grid columnconfigure .fr 0 -weight 1
    wm title . "Tcl3D demo: Nopper's core profile tutorials (Example 04 - Phong rendering of a sphere)"

    # Watch for Esc key and Quit messages
    wm protocol . WM_DELETE_WINDOW "ExitProg"
    bind . <Key-Escape> "ExitProg"

    .fr.usage insert end "Key-Escape Exit"
    if { [info exists msgStr] } {
        .fr.usage insert end $msgStr
        .fr.usage itemconfigure end -background red
    } else {
        .fr.usage configure -state disabled
    }
}

CreateWindow
ReshapeCallback .fr.toglwin

PrintInfo [tcl3dOglGetInfoString]
