# Lesson45.tcl
#
# Paul Frazee's Vertex Buffer Object Tutorial
#
# Code Commmenting And Clean Up By Jeff Molofee ( NeHe )
# If You've Found This Code Useful, Please Let Me Know.
# Visit My Site At nehe.gamedev.net
#
# Modified for Tcl3D by Paul Obermeier 2006/08/17
# See www.tcl3d.org for the Tcl3D extension.

package require Img
package require tcl3d

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Determine the directory of this script.
set gDemo(scriptDir) [file dirname [info script]]

# Display mode.
set fullScreen false

# Window size.
set gDemo(winWidth)  640
set gDemo(winHeight) 480

# Mesh Generation Paramaters
set MESH_RESOLUTION 4.0         ; # Pixels Per Vertex
set MESH_HEIGHTSCALE 1.0        ; # Mesh Height Scale

set g_bUseVBOs 1                ; # If 1, VBOs Will Be Forced On

set g_nTextureId [tcl3dVector GLuint 1] ; # Texture ID

# Vertex Buffer Object Names
set g_nVBOVertices  [tcl3dVector GLuint 1] ; # Vertex VBO Name
set g_nVBOTexCoords [tcl3dVector GLuint 1] ; # Texture Coordinate VBO Name

set g_fVBOSupported false   ; # ARB_vertex_buffer_object supported?
set g_flYRot 0.0            ; # Rotation

set stopwatch [tcl3dNewSwatch]
set FPS_UPDATE_FREQUENCY 50
set frameCount 0
set totalFrames 0

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    ExitProg
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# Print info message into widget at the bottom of the window.
proc PrintTimeInfo { msg } {
    if { [winfo exists .fr.timeinfo] } {
        .fr.timeinfo configure -text $msg
    }
}

proc SetFullScreenMode { win } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win $sw $sh
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+0" $sw $sh]
    wm geometry $win $fmtStr
    wm overrideredirect $win 1
    focus -force $win
}

proc SetWindowMode { win w h } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win 10 10
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+25" $w $h]
    wm geometry $win $fmtStr
    wm overrideredirect $win 0
    focus -force $win
}

# Toggle between windowing and fullscreen mode.
proc ToggleWindowMode {} {
    if { $::fullScreen } {
        SetFullScreenMode .
        set ::fullScreen false
    } else {
        SetWindowMode . $::gDemo(winWidth) $::gDemo(winHeight)
        set ::fullScreen true
    }
}

proc GetElapsedSeconds {} {
    set currentTime [tcl3dLookupSwatch $::stopwatch]
    set sec [expr $currentTime - $::elapsedLastTime]
    set ::elapsedLastTime $currentTime
    return $sec
}

proc GetFPS { { elapsedFrames 1 } } {
    set ::totalFrames [expr $::totalFrames + $elapsedFrames]
    set currentTime [tcl3dLookupSwatch $::stopwatch]
    set fps [expr $elapsedFrames / ($currentTime - $::s_lastTime)]
    set ::s_lastTime $currentTime
    return $fps
}

proc DisplayFPS {} {
    global frameCount

    if { ! [info exists ::animateId] } {
        # Animation has been stopped. Do not update fps measurement.
        return
    }
    incr frameCount
    if { $frameCount == $::FPS_UPDATE_FREQUENCY } {
        set msg [format "%d triangles, %.0f fps" \
             [expr $::g_nVertexCount / 3] [GetFPS $frameCount]]
        if { $::g_fVBOSupported } {
            append msg ", using VBOs"
        } else {
            append msg ", not using VBOs"
        }
        PrintTimeInfo $msg 
        set frameCount 0
    }
}

proc LoadImage { imgName numChans } {
    if { $numChans != 3 && $numChans != 4 } {
        error "Error: Only 3 or 4 channels allowed ($numChans supplied)"
    }
    set texName [file join $::gDemo(scriptDir) $imgName]
    set retVal [catch {set phImg [image create photo -file $texName]} err1]
    if { $retVal != 0 } {
        error "Error reading image $texName ($err1)"
    } else {
        set w [image width  $phImg]
        set h [image height $phImg]
        set texImg [tcl3dVectorFromPhoto $phImg $numChans]
    }
    return [list $phImg $texImg $w $h]
}

proc LoadHeightmap { fileName flHeightScale flResolution } {
    # Load Texture Data
    set imgInfo [LoadImage $fileName 3]
    set phImg     [lindex $imgInfo 0]
    set imgData   [lindex $imgInfo 1]
    set imgWidth  [lindex $imgInfo 2]
    set imgHeight [lindex $imgInfo 3]

    # Generate Vertex Field
    set vectorList [tcl3dDemoUtilHeightmapFromPhoto $phImg \
                    $flHeightScale $flResolution]
    set ::g_pVertices    [lindex $vectorList 0]
    set ::g_pTexCoords   [lindex $vectorList 1]
    set ::g_nVertexCount [lindex $vectorList 2]

    # Load The Texture Into OpenGL
    glGenTextures 1 $::g_nTextureId ; # Get An Open ID
    glBindTexture GL_TEXTURE_2D [$::g_nTextureId get 0] ; # Bind The Texture
    glTexImage2D GL_TEXTURE_2D 0 3 $imgWidth $imgHeight 0 GL_RGB GL_UNSIGNED_BYTE $imgData
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MIN_FILTER $::GL_LINEAR
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MAG_FILTER $::GL_LINEAR

    $imgData delete
    image delete $phImg
}

proc BuildVBOs {} {
    # Generate And Bind The Vertex Buffer
    glGenBuffersARB 1 $::g_nVBOVertices ; # Get A Valid Name
    glBindBufferARB GL_ARRAY_BUFFER_ARB [$::g_nVBOVertices get 0] ; # Bind The Buffer
    # Load The Data
    glBufferDataARB GL_ARRAY_BUFFER_ARB [expr $::g_nVertexCount*3*4] \
                    $::g_pVertices GL_STATIC_DRAW_ARB
    set retVal [tcl3dOglGetError]
    if { $retVal ne "" } {
        error "Error creating VBO for vertices: $retVal"
    }

    # Generate And Bind The Texture Coordinate Buffer
    glGenBuffersARB 1 $::g_nVBOTexCoords ; # Get A Valid Name
    glBindBufferARB GL_ARRAY_BUFFER_ARB [$::g_nVBOTexCoords get 0] ; # Bind The Buffer
    # Load The Data
    glBufferDataARB GL_ARRAY_BUFFER_ARB [expr $::g_nVertexCount*2*4] \
                    $::g_pTexCoords GL_STATIC_DRAW_ARB
    set retVal [tcl3dOglGetError]
    if { $retVal ne "" } {
        error "Error creating VBO for tex coords: $retVal"
    }

    # Our Copy Of The Data Is No Longer Necessary, It Is Safe In The Graphics Card
    $::g_pVertices  delete
    $::g_pTexCoords delete
}

# Resize And Initialize The GL Window
proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h                ; # Reset The Current Viewport
    glMatrixMode GL_PROJECTION          ; # Select The Projection Matrix
    glLoadIdentity                      ; # Reset The Projection Matrix

    # Calculate The Aspect Ratio Of The Window
    gluPerspective 45.0 [expr double($w)/double($h)] 1 1000.0

    glMatrixMode GL_MODELVIEW   ; # Select The Modelview Matrix
    glLoadIdentity              ; # Reset The Modelview Matrix
    set ::gDemo(winWidth)  $w
    set ::gDemo(winHeight) $h
}

proc InitHeightData { toglwin } {
    LoadHeightmap "Terrain.bmp" $::MESH_HEIGHTSCALE $::MESH_RESOLUTION

    # Check For VBOs Supported
    if { $::g_bUseVBOs } {
        set ::g_fVBOSupported [tcl3dOglHaveExtension $toglwin "GL_ARB_vertex_buffer_object"]
        if { $::g_fVBOSupported } {
            # Load Vertex Data Into The Graphics Card Memory
            BuildVBOs   ; # Build The VBOs
        }
    } else {
        set ::g_fVBOSupported false
    }
}

# All Setup For OpenGL Goes Here
proc CreateCallback { toglwin } {
    tcl3dStartSwatch $::stopwatch

    InitHeightData $toglwin

    # Setup GL States
    glClearColor 0.0 0.0 0.0 0.5    ; # Black Background
    glClearDepth 1.0                ; # Depth Buffer Setup
    glDepthFunc GL_LEQUAL           ; # The Type Of Depth Testing (Less Or Equal)
    glEnable GL_DEPTH_TEST          ; # Enable Depth Testing
    glShadeModel GL_SMOOTH          ; # Select Smooth Shading
    glEnable GL_TEXTURE_2D          ; # Enable Textures
    glColor4f 1.0 1.0 1.0 1.0       ; # Set The Color To White
    # Set Perspective Calculations To Most Accurate
    glHint GL_PERSPECTIVE_CORRECTION_HINT GL_NICEST

    tcl3dResetSwatch $::stopwatch
    set ::startTime [tcl3dLookupSwatch $::stopwatch]
    set ::s_lastTime $::startTime
    set ::elapsedLastTime $::startTime
}

# Here's Where We Do All The Drawing
proc DisplayCallback { toglwin } {
    set ::g_flYRot [expr $::g_flYRot + [GetElapsedSeconds] * 25.0] ; # Consistantly Rotate The Scenery

    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glLoadIdentity          ; # Reset The Modelview Matrix

    DisplayFPS

    # Move The Camera
    glTranslatef 0.0 -220.0 0.0         ; # Move Above The Terrain
    glRotatef 10.0 1.0 0.0 0.0          ; # Look Down Slightly
    glRotatef $::g_flYRot 0.0 1.0 0.0   ; # Rotate The Camera

    # Enable Pointers
    glEnableClientState GL_VERTEX_ARRAY         ; # Enable Vertex Arrays
    glEnableClientState GL_TEXTURE_COORD_ARRAY  ; # Enable Texture Coord Arrays

    # Set Pointers To Our Data
    if { $::g_fVBOSupported } {
        glBindBufferARB GL_ARRAY_BUFFER_ARB [$::g_nVBOVertices get 0]
        # Set The Vertex Pointer To The Vertex Buffer
        glVertexPointer 3 GL_FLOAT 0 NULL
        glBindBufferARB GL_ARRAY_BUFFER_ARB [$::g_nVBOTexCoords get 0]
        # Set The TexCoord Pointer To The TexCoord Buffer
        glTexCoordPointer 2 GL_FLOAT 0 NULL 
    } else {
        # Set The Vertex Pointer To Our Vertex Data
        glVertexPointer 3 GL_FLOAT 0 $::g_pVertices 
        # Set The Vertex Pointer To Our TexCoord Data
        glTexCoordPointer 2 GL_FLOAT 0 $::g_pTexCoords
    }

    # Draw All Of The Triangles At Once
    glDrawArrays GL_TRIANGLES 0 $::g_nVertexCount

    # Disable Pointers
    glDisableClientState GL_VERTEX_ARRAY        ; # Disable Vertex Arrays
    glDisableClientState GL_TEXTURE_COORD_ARRAY ; # Disable Texture Coord Arrays

    $toglwin swapbuffers
}

proc Animate {} {
    .fr.toglwin postredisplay
    set ::animateId [tcl3dAfterIdle Animate]
}

proc StartAnimation {} {
    tcl3dStartSwatch $::stopwatch
    if { ! [info exists ::animateId] } {
        Animate
    }
}

proc StopAnimation {} {
    tcl3dStopSwatch $::stopwatch
    if { [info exists ::animateId] } {
        after cancel $::animateId 
        unset ::animateId
    }
}

proc Cleanup {} {
    # Delete VBOs
    if { $::g_fVBOSupported } {
        glDeleteBuffersARB 1 [$::g_nVBOVertices get 0]
        glDeleteBuffersARB 1 [$::g_nVBOTexCoords get 0]
    }
}

# Put all exit related code here.
proc ExitProg {} {
    exit
}

# Create the OpenGL window and some Tk helper widgets.
proc CreateWindow {} {
    frame .fr
    pack .fr -expand 1 -fill both
    # Create Our OpenGL Window
    togl .fr.toglwin -width $::gDemo(winWidth) -height $::gDemo(winHeight) \
                     -double true -depth true \
                     -createcommand CreateCallback \
                     -reshapecommand ReshapeCallback \
                     -displaycommand DisplayCallback 
    listbox .fr.usage -font $::listFont -height 3
    label   .fr.timeinfo -bg white
    label   .fr.info
    grid .fr.toglwin  -row 0 -column 0 -sticky news
    grid .fr.usage    -row 1 -column 0 -sticky news
    grid .fr.timeinfo -row 2 -column 0 -sticky news
    grid .fr.info     -row 3 -column 0 -sticky news
    grid rowconfigure .fr 0 -weight 1
    grid columnconfigure .fr 0 -weight 1
    wm title . "Tcl3D demo: Paul Frazee's Vertex Buffer Object Tutorial (Lesson 45)"

    # Watch For ESC Key And Quit Messages
    wm protocol . WM_DELETE_WINDOW "ExitProg"
    bind . <Key-Escape> "ExitProg"
    bind . <Key-F1>     "ToggleWindowMode"

    bind .fr.toglwin <1> "StartAnimation"
    bind .fr.toglwin <2> "StopAnimation"
    bind .fr.toglwin <3> "StopAnimation"
    bind .fr.toglwin <Control-Button-1> "StopAnimation"

    .fr.usage insert end "Key-Escape Exit"
    .fr.usage insert end "Key-F1     Toggle window mode"
    .fr.usage insert end "Mouse-L|MR Start|Stop animation"

    .fr.usage configure -state disabled
}

if { $argc >= 1 } {
    if { [string compare -nocase [lindex $argv 0] "vbo"] == 0 } {
        set ::g_bUseVBOs 1
    } elseif { [string compare -nocase [lindex $argv 0] "novbo"] == 0 } {
        set ::g_bUseVBOs 0
    }
}

CreateWindow
PrintInfo [tcl3dOglGetInfoString]
if { [file tail [info script]] eq [file tail $::argv0] } {
    # If started directly from tclsh or wish, then start animation.
    update
    StartAnimation
}
