# Lesson41.tcl
#
# NeHe's Volumetric Fog Tutorial
#
# This Code Was Created By Jeff Molofee 2003
# If You've Found This Code Useful, Please Let Me Know.
# Visit My Site At nehe.gamedev.net
#
# Modified for Tcl3D by Paul Obermeier 2006/08/27
# See www.tcl3d.org for the Tcl3D extension.

package require Img
package require tcl3d

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Determine the directory of this script.
set gDemo(scriptDir) [file dirname [info script]]

# Display mode.
set fullScreen false

# Window size.
set gDemo(winWidth)  640
set gDemo(winHeight) 480

set camz     0.0                    ; # Camera Z Depth
set fogColor { 0.6 0.3 0.0 1.0 }    ; # Fog Colour Light
set texture  [tcl3dVector GLuint 1] ; # One Texture (For The Walls)

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    ExitProg
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc SetFullScreenMode { win } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win $sw $sh
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+0" $sw $sh]
    wm geometry $win $fmtStr
    wm overrideredirect $win 1
    focus -force $win
}

proc SetWindowMode { win w h } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win 10 10
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+25" $w $h]
    wm geometry $win $fmtStr
    wm overrideredirect $win 0
    focus -force $win
}

# Toggle between windowing and fullscreen mode.
proc ToggleWindowMode {} {
    if { $::fullScreen } {
        SetFullScreenMode .
        set ::fullScreen false
    } else {
        SetWindowMode . $::gDemo(winWidth) $::gDemo(winHeight)
        set ::fullScreen true
    }
}

proc SetDepth { val } {
    set ::camz [expr $::camz + $val]
    if { $::camz > 14.0 } {
        set ::camz 14.0
    } elseif { $::camz < -19.0 } {
        set ::camz -19.0
    }
    .fr.toglwin postredisplay
}

proc LoadImage { imgName numChans } {
    if { $numChans != 3 && $numChans != 4 } {
        error "Error: Only 3 or 4 channels allowed ($numChans supplied)"
    }
    set texName [file join $::gDemo(scriptDir) "Data" $imgName]
    set retVal [catch {set phImg [image create photo -file $texName]} err1]
    if { $retVal != 0 } {
        error "Error reading image $texName ($err1)"
    } else {
        set w [image width  $phImg]
        set h [image height $phImg]
        set texImg [tcl3dVectorFromPhoto $phImg $numChans]
        image delete $phImg
    }
    return [list $texImg $w $h]
}

# Load Bitmaps And Convert To Textures
proc LoadGLTextures {} {

    # Load The Wall Texture
    set imgInfo [LoadImage "Wall.bmp" 4]
    set imgData   [lindex $imgInfo 0]
    set imgWidth  [lindex $imgInfo 1]
    set imgHeight [lindex $imgInfo 2]

    # Create The Texture
    glGenTextures 1 $::texture  

    glBindTexture GL_TEXTURE_2D [$::texture get 0]
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MIN_FILTER $::GL_LINEAR
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MAG_FILTER $::GL_LINEAR
    glTexImage2D GL_TEXTURE_2D 0 3 $imgWidth $imgHeight \
                 0 GL_RGBA GL_UNSIGNED_BYTE $imgData

    # Delete the image data vector.
    $imgData delete
}

proc ExtensionInit { toglwin } {
    if { ![tcl3dOglHaveExtension $toglwin "GL_EXT_fog_coord"] } {
        error "GL_EXT_fog_coord missing"
    }
}

# Resize And Initialize The GL Window
proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h                ; # Reset The Current Viewport
    glMatrixMode GL_PROJECTION          ; # Select The Projection Matrix
    glLoadIdentity                      ; # Reset The Projection Matrix

    # Calculate The Aspect Ratio Of The Window
    gluPerspective 45.0 [expr double($w)/double($h)] 0.1 100.0

    glMatrixMode GL_MODELVIEW           ; # Select The Modelview Matrix
    glLoadIdentity                      ; # Reset The Modelview Matrix
    set ::gDemo(winWidth)  $w
    set ::gDemo(winHeight) $h
}

# All Setup For OpenGL Goes Here
proc CreateCallback { toglwin } {
    ExtensionInit $toglwin                  ; # Check If Fog Extension Is Available

    LoadGLTextures                          ; # Load The Wall Texture
    
    glEnable GL_TEXTURE_2D                  ; # Enable 2D Texture Mapping
    glClearColor 0.0 0.0 0.0 0.5            ; # Black Background
    glClearDepth 1.0                        ; # Depth Buffer Setup
    glDepthFunc GL_LEQUAL                   ; # The Type Of Depth Testing To Do
    glEnable GL_DEPTH_TEST                  ; # Enables Depth Testing
    glShadeModel GL_SMOOTH                  ; # Enable Smooth Shading
    glHint GL_PERSPECTIVE_CORRECTION_HINT \
           GL_NICEST                        ; # Really Nice Perspective Calculations

    # Set Up Fog 
    glEnable GL_FOG                         ; # Enable Fog
    glFogi GL_FOG_MODE $::GL_LINEAR         ; # Fog Fade Is Linear
    glFogfv GL_FOG_COLOR $::fogColor        ; # Set The Fog Color
    glFogf GL_FOG_START 0.0                 ; # Set The Fog Start
    glFogf GL_FOG_END   1.0                 ; # Set The Fog End
    glHint GL_FOG_HINT GL_NICEST            ; # Per-Pixel Fog Calculation
    glFogi GL_FOG_COORDINATE_SOURCE_EXT \
           $::GL_FOG_COORDINATE_EXT         ; # Set Fog Based On Vertice Coordinates

    set ::camz -19.0                        ; # Set Camera Z Position To -19.0
}

# Here's Where We Do All The Drawing
proc DisplayCallback { toglwin } {
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glLoadIdentity         ; # Reset The Modelview Matrix

    glTranslatef 0.0 0.0 $::camz

    glBegin GL_QUADS       ; # Back Wall
        glFogCoordfEXT 1.0 ;    glTexCoord2f 0.0 0.0 ;  glVertex3f -2.5 -2.5 -15.0
        glFogCoordfEXT 1.0 ;    glTexCoord2f 1.0 0.0 ;  glVertex3f  2.5 -2.5 -15.0
        glFogCoordfEXT 1.0 ;    glTexCoord2f 1.0 1.0 ;  glVertex3f  2.5  2.5 -15.0
        glFogCoordfEXT 1.0 ;    glTexCoord2f 0.0 1.0 ;  glVertex3f -2.5  2.5 -15.0
    glEnd

    glBegin GL_QUADS       ; # Floor
        glFogCoordfEXT 1.0 ;    glTexCoord2f 0.0 0.0 ;  glVertex3f -2.5 -2.5 -15.0
        glFogCoordfEXT 1.0 ;    glTexCoord2f 1.0 0.0 ;  glVertex3f  2.5 -2.5 -15.0
        glFogCoordfEXT 0.0 ;    glTexCoord2f 1.0 1.0 ;  glVertex3f  2.5 -2.5  15.0
        glFogCoordfEXT 0.0 ;    glTexCoord2f 0.0 1.0 ;  glVertex3f -2.5 -2.5  15.0
    glEnd

    glBegin GL_QUADS       ; # Roof
        glFogCoordfEXT 1.0 ;    glTexCoord2f 0.0 0.0 ;  glVertex3f -2.5  2.5 -15.0
        glFogCoordfEXT 1.0 ;    glTexCoord2f 1.0 0.0 ;  glVertex3f  2.5  2.5 -15.0
        glFogCoordfEXT 0.0 ;    glTexCoord2f 1.0 1.0 ;  glVertex3f  2.5  2.5  15.0
        glFogCoordfEXT 0.0 ;    glTexCoord2f 0.0 1.0 ;  glVertex3f -2.5  2.5  15.0
    glEnd

    glBegin GL_QUADS       ; # Right Wall
        glFogCoordfEXT 0.0 ;    glTexCoord2f 0.0 0.0 ;  glVertex3f  2.5 -2.5  15.0
        glFogCoordfEXT 0.0 ;    glTexCoord2f 0.0 1.0 ;  glVertex3f  2.5  2.5  15.0
        glFogCoordfEXT 1.0 ;    glTexCoord2f 1.0 1.0 ;  glVertex3f  2.5  2.5 -15.0
        glFogCoordfEXT 1.0 ;    glTexCoord2f 1.0 0.0 ;  glVertex3f  2.5 -2.5 -15.0
    glEnd

    glBegin GL_QUADS       ; # Left Wall
        glFogCoordfEXT 0.0 ;    glTexCoord2f 0.0 0.0 ;  glVertex3f -2.5 -2.5  15.0
        glFogCoordfEXT 0.0 ;    glTexCoord2f 0.0 1.0 ;  glVertex3f -2.5  2.5  15.0
        glFogCoordfEXT 1.0 ;    glTexCoord2f 1.0 1.0 ;  glVertex3f -2.5  2.5 -15.0
        glFogCoordfEXT 1.0 ;    glTexCoord2f 1.0 0.0 ;  glVertex3f -2.5 -2.5 -15.0
    glEnd
 
    glFlush                ; # Flush The GL Rendering Pipeline

    $toglwin swapbuffers
}

# Put all exit related code here.
proc ExitProg {} {
    exit
}

# Create the OpenGL window and some Tk helper widgets.
proc CreateWindow {} {
    frame .fr
    pack .fr -expand 1 -fill both
    # Create Our OpenGL Window
    togl .fr.toglwin -width $::gDemo(winWidth) -height $::gDemo(winHeight) \
                     -double true -depth true -alpha true \
                     -createcommand CreateCallback \
                     -reshapecommand ReshapeCallback \
                     -displaycommand DisplayCallback 
    listbox .fr.usage -font $::listFont -height 3
    label   .fr.info
    grid .fr.toglwin -row 0 -column 0 -sticky news
    grid .fr.usage   -row 1 -column 0 -sticky news
    grid .fr.info    -row 2 -column 0 -sticky news
    grid rowconfigure .fr 0 -weight 1
    grid columnconfigure .fr 0 -weight 1
    wm title . "Tcl3D demo: NeHe's Volumetric Fog Tutorial (Lesson 41)"

    # Watch For ESC Key And Quit Messages
    wm protocol . WM_DELETE_WINDOW "ExitProg"
    bind . <Key-Escape> "ExitProg"
    bind . <Key-F1>     "ToggleWindowMode"
    bind . <Key-Up>     "SetDepth  0.25"
    bind . <Key-Down>   "SetDepth -0.25"

    .fr.usage insert end "Key-Escape  Exit"
    .fr.usage insert end "Key-F1      Toggle window mode"
    .fr.usage insert end "Key-Up|Down Move object closer|further"

    .fr.usage configure -state disabled
}

CreateWindow
PrintInfo [tcl3dOglGetInfoString]
