# Lesson33.tcl
#
# NeHe & Evan 'terminate' Pipho's TGA Loading Tutorial
#
# Loading Uncompressed and Compressed .TGA Files with the Img extension.
#
# This Code Was Created By Evan Pipho
# If You've Found This Code Useful, Please Let Me Know.
# Visit My Site At nehe.gamedev.net
#
# Modified for Tcl3D by Paul Obermeier 2006/08/16
# See www.tcl3d.org for the Tcl3D extension.

package require Img
package require tcl3d

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Determine the directory of this script.
set gDemo(scriptDir) [file dirname [info script]]

# Display mode.
set fullScreen false

# Window size.
set gDemo(winWidth)  640
set gDemo(winHeight) 480

set spin 0.0                            ; # Spin Variable

set texture [tcl3dVector GLuint 2]      ; # Storage For Two Textures

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    ExitProg
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc SetFullScreenMode { win } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win $sw $sh
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+0" $sw $sh]
    wm geometry $win $fmtStr
    wm overrideredirect $win 1
    focus -force $win
}

proc SetWindowMode { win w h } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win 10 10
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+25" $w $h]
    wm geometry $win $fmtStr
    wm overrideredirect $win 0
    focus -force $win
}

# Toggle between windowing and fullscreen mode.
proc ToggleWindowMode {} {
    if { $::fullScreen } {
        SetFullScreenMode .
        set ::fullScreen false
    } else {
        SetWindowMode . $::gDemo(winWidth) $::gDemo(winHeight)
        set ::fullScreen true
    }
}

proc LoadImage { imgName numChans } {
    if { $numChans != 3 && $numChans != 4 } {
        error "Error: Only 3 or 4 channels allowed ($numChans supplied)"
    }
    set texName [file join $::gDemo(scriptDir) "Data" $imgName]
    set retVal [catch {set phImg [image create photo -file $texName]} err1]
    if { $retVal != 0 } {
        error "Error reading image $texName ($err1)"
    } else {
        set w [image width  $phImg]
        set h [image height $phImg]
        set texImg [tcl3dVectorFromPhoto $phImg $numChans]
        image delete $phImg
    }
    return [list $texImg $w $h]
}

proc LoadGLTextures {} {
    # Load uncompressed texture image.
    set imgInfo [LoadImage "Uncompressed.tga" 3]
    set imgData(0)   [lindex $imgInfo 0]
    set imgWidth(0)  [lindex $imgInfo 1]
    set imgHeight(0) [lindex $imgInfo 2]

    # Load compressed texture image.
    set imgInfo [LoadImage "Compressed.tga" 3]
    set imgData(1)   [lindex $imgInfo 0]
    set imgWidth(1)  [lindex $imgInfo 1]
    set imgHeight(1) [lindex $imgInfo 2]

    # Create The Textures
    glGenTextures 2 $::texture 
    for { set i 0 } { $i < 2 } { incr i } {
        glBindTexture GL_TEXTURE_2D [$::texture get $i]
        glTexImage2D GL_TEXTURE_2D 0 3 $imgWidth($i) $imgHeight($i) \
                     0 GL_RGB GL_UNSIGNED_BYTE $imgData($i)
        glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MIN_FILTER $::GL_LINEAR
        glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MAG_FILTER $::GL_LINEAR
    }

    # Delete the image data vectors.
    for { set i 0 } { $i < 2 } { incr i } {
        $imgData($i) delete
    }
}

# Resize And Initialize The GL Window
proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h        ; # Reset The Current Viewport
    glMatrixMode GL_PROJECTION  ; # Select The Projection Matrix
    glLoadIdentity              ; # Reset The Projection Matrix

    # Calculate The Aspect Ratio Of The Window
    gluPerspective 45.0 [expr double($w)/double($h)] 0.1 100.0

    glMatrixMode GL_MODELVIEW   ; # Select The Modelview Matrix
    glLoadIdentity              ; # Reset The Modelview Matrix
    set ::gDemo(winWidth)  $w
    set ::gDemo(winHeight) $h
}

# All Setup For OpenGL Goes Here
proc CreateCallback { toglwin } {
    LoadGLTextures                          ; # Jump To Texture Loading Routine
    glEnable GL_TEXTURE_2D                  ; # Enable Texture Mapping

    glShadeModel GL_SMOOTH                  ; # Enable Smooth Shading
    glClearColor 0.0 0.0 0.0 0.5            ; # Black Background
    glClearDepth 1.0                        ; # Depth Buffer Setup
    glEnable GL_DEPTH_TEST                  ; # Enables Depth Testing
    glDepthFunc GL_LEQUAL                   ; # The Type Of Depth Testing To Do
    glHint GL_PERSPECTIVE_CORRECTION_HINT GL_NICEST ; # Really Nice Perspective Calculations
}

# Here's Where We Do All The Drawing
proc DisplayCallback { toglwin } {
    # Clear Screen And Depth Buffer
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT] 

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glLoadIdentity                      ; # Reset The Current Modelview Matrix
    glTranslatef 0.0 0.0 -10.0

    if { [info exists ::animateId] } {
        set ::spin [expr {$::spin + 0.05}]
    }

    for { set loop 0 } { $loop < 20 } { incr loop } {
        glPushMatrix
        glRotatef [expr {$::spin+$loop*18.0}] 1.0 0.0 0.0 ; # Rotate On The X-Axis (Up - Down)
        glTranslatef -2.0 2.0 0.0

        glBindTexture GL_TEXTURE_2D [$::texture get 0]
        glBegin GL_QUADS
            glTexCoord2f 0.0 1.0 ; glVertex3f -1.0  1.0 0.0
            glTexCoord2f 1.0 1.0 ; glVertex3f  1.0  1.0 0.0
            glTexCoord2f 1.0 0.0 ; glVertex3f  1.0 -1.0 0.0
            glTexCoord2f 0.0 0.0 ; glVertex3f -1.0 -1.0 0.0
        glEnd
        glPopMatrix

        glPushMatrix
        glTranslatef 2.0 0.0 0.0                          ; # Translate 2 Units To The Right
        glRotatef [expr {$::spin+$loop*36.0}] 0.0 1.0 0.0 ; # Rotate On The Y-Axis (Left - Right)
        glTranslatef 1.0 0.0 0.0                          ; # Move One Unit Right

        glBindTexture GL_TEXTURE_2D [$::texture get 1]
        glBegin GL_QUADS
            glTexCoord2f 0.0 1.0 ; glVertex3f -1.0  1.0 0.0
            glTexCoord2f 1.0 1.0 ; glVertex3f  1.0  1.0 0.0
            glTexCoord2f 1.0 0.0 ; glVertex3f  1.0 -1.0 0.0
            glTexCoord2f 0.0 0.0 ; glVertex3f -1.0 -1.0 0.0
        glEnd
        glPopMatrix
    }
    $toglwin swapbuffers
}

proc Animate {} {
    .fr.toglwin postredisplay
    set ::animateId [tcl3dAfterIdle Animate]
}

proc StartAnimation {} {
    if { ! [info exists ::animateId] } {
        Animate
    }
}

proc StopAnimation {} {
    if { [info exists ::animateId] } {
        after cancel $::animateId 
        unset ::animateId
    }
}

# Put all exit related code here.
proc ExitProg {} {
    exit
}

# Create the OpenGL window and some Tk helper widgets.
proc CreateWindow {} {
    frame .fr
    pack .fr -expand 1 -fill both
    # Create Our OpenGL Window
    togl .fr.toglwin -width $::gDemo(winWidth) -height $::gDemo(winHeight) \
                     -double true -depth true \
                     -createcommand CreateCallback \
                     -reshapecommand ReshapeCallback \
                     -displaycommand DisplayCallback 
    listbox .fr.usage -font $::listFont -height 3
    label   .fr.info
    grid .fr.toglwin -row 0 -column 0 -sticky news
    grid .fr.usage   -row 1 -column 0 -sticky news
    grid .fr.info    -row 2 -column 0 -sticky news
    grid rowconfigure .fr 0 -weight 1
    grid columnconfigure .fr 0 -weight 1
    wm title . "Tcl3D demo: NeHe & Evan 'terminate' Pipho's TGA Loading Tutorial (Lesson 33)"

    # Watch For ESC Key And Quit Messages
    wm protocol . WM_DELETE_WINDOW "ExitProg"
    bind . <Key-Escape> "ExitProg"
    bind . <Key-F1>     "ToggleWindowMode"

    bind .fr.toglwin <1> "StartAnimation"
    bind .fr.toglwin <2> "StopAnimation"
    bind .fr.toglwin <3> "StopAnimation"
    bind .fr.toglwin <Control-Button-1> "StopAnimation"

    .fr.usage insert end "Key-Escape Exit"
    .fr.usage insert end "Key-F1     Toggle window mode"
    .fr.usage insert end "Mouse-L|MR Start|Stop animation"

    .fr.usage configure -state disabled
}

CreateWindow
PrintInfo [tcl3dOglGetInfoString]

if { [file tail [info script]] eq [file tail $::argv0] } {
    # If started directly from tclsh or wish, then start animation.
    update
    StartAnimation
}
