# Lesson25.tcl
#
# Piotr Cieslak & NeHe's Morphing Points Tutorial
#
# This Code Was Created By Pet & Commented/Cleaned Up By Jeff Molofee
# If You've Found This Code Useful, Please Let Me Know.
# Visit NeHe Productions At http://nehe.gamedev.net
#
# Modified for Tcl3D by Paul Obermeier 2007/03/03
# See www.tcl3d.org for the Tcl3D extension.

package require tcl3d

# Font to be used in the Tk listbox.
set gDemo(listFont) {-family {Courier} -size 10}

# Determine the directory of this script.
set gDemo(scriptDir) [file dirname [info script]]

# Display mode.
set gDemo(fullScreen) true

# Window size.
set gDemo(winWidth)  640
set gDemo(winHeight) 480

set gDemo(xrot)    0.0  ; # X Rotation
set gDemo(yrot)    0.0  ; # Y Rotation
set gDemo(zrot)    0.0  ; # Z Rotation
set gDemo(xspeed)  0.1  ; # X Spin Speed
set gDemo(yspeed)  0.1  ; # Y Spin Speed
set gDemo(zspeed)  0.0  ; # Z Spin Speed
set gDemo(cx)      0.0  ; # X Position
set gDemo(cy)      0.0  ; # Y Position
set gDemo(cz)    -15.0  ; # Z Position

set gDemo(key)     1    ; # Used To Make Sure Same Morph Key Is Not Pressed

set gDemo(step)    0    ; # Step Counter
set gDemo(steps) 200    ; # Maximum Number Of Steps
set gDemo(morph) false  ; # Default morph To False (Not Morphing)

set gDemo(maxver) 0     ; # Will Eventually Hold The Maximum Number Of Vertices

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    ExitProg
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc SetFullScreenMode { win } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win $sw $sh
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+0" $sw $sh]
    wm geometry $win $fmtStr
    wm overrideredirect $win 1
    focus -force $win
}

proc SetWindowMode { win w h } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win 10 10
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+25" $w $h]
    wm geometry $win $fmtStr
    wm overrideredirect $win 0
    focus -force $win
}

# Toggle between windowing and fullscreen mode.
proc ToggleWindowMode {} {
    if { $::gDemo(fullScreen) } {
        SetFullScreenMode .
        set ::gDemo(fullScreen) false
    } else {
        SetWindowMode . $::gDemo(winWidth) $::gDemo(winHeight)
        set ::gDemo(fullScreen) true
    }
}

proc Reset {} {
    global gDemo

    set gDemo(xrot)    0.0
    set gDemo(yrot)    0.0
    set gDemo(zrot)    0.0
    set gDemo(xspeed)  0.0
    set gDemo(yspeed)  0.0
    set gDemo(zspeed)  0.0
    set gDemo(cx)      0.0
    set gDemo(cy)      0.0
    set gDemo(cz)    -15.0
    .fr.toglwin postredisplay
}

# Set z speed.
proc SetZSpeed { val } {
    set ::gDemo(zspeed) [expr {$::gDemo(zspeed) + $val}]
    .fr.toglwin postredisplay
}

# Set y speed.
proc SetYSpeed { val } {
    set ::gDemo(yspeed) [expr {$::gDemo(yspeed) + $val}]
    .fr.toglwin postredisplay
}

# Set x speed.
proc SetXSpeed { val } {
    set ::gDemo(xspeed) [expr {$::gDemo(xspeed) + $val}]
    .fr.toglwin postredisplay
}

# Set z position.
proc SetZPos { val } {
    set ::gDemo(cz) [expr {$::gDemo(cz) + $val}]
    .fr.toglwin postredisplay
}

# Set y position.
proc SetYPos { val } {
    set ::gDemo(cy) [expr {$::gDemo(cy) + $val}]
    .fr.toglwin postredisplay
}

# Set x position.
proc SetXPos { val } {
    set ::gDemo(cx) [expr {$::gDemo(cx) + $val}]
    .fr.toglwin postredisplay
}

# Switch between morphing algorithms.
proc SwitchMorph { key } {
    global gDemo

    if { $key == 1 && $gDemo(key) != 1 && !$gDemo(morph) } {
        set gDemo(key) 1
        set gDemo(morph) true
        set gDemo(dest) 1
    }
    if { $key == 2 && $gDemo(key) != 2 && !$gDemo(morph) } {
        set gDemo(key) 2
        set gDemo(morph) true
        set gDemo(dest) 2
    }
    if { $key == 3 && $gDemo(key) != 3 && !$gDemo(morph) } {
        set gDemo(key) 3
        set gDemo(morph) true
        set gDemo(dest) 3
    }
    if { $key == 4 && $gDemo(key) != 4 && !$gDemo(morph) } {
        set gDemo(key) 4
        set gDemo(morph) true
        set gDemo(dest) 4
    }
}

proc readstr { fp } {
    while { 1 } {
        gets $fp line
        # Empty line or comment: Read next line
        if { ($line eq "") || ([string index $line 0] eq "/") } {
            continue
        } else {
            break
        }
    }
    return $line
}

# Loads Object Number k From File (name)
proc objload {name k } {
    global gObjects gDemo

    set fileName [file join $::gDemo(scriptDir) "Data" $name]
    set filein [open $fileName r]

    set oneline [readstr $filein]
    scan $oneline "Vertices: %d" ver
    set gObjects($k,verts) $ver

    for { set i 0 } { $i < $ver } { incr i } {
        set oneline [readstr $filein]
        scan $oneline "%f %f %f" rx ry rz
        lappend gObjects($k,points) [list $rx $ry $rz]
    }
    close $filein

    # Keeps Track Of Highest Number Of Vertices Used In Any Of The Objects
    if { $ver > $gDemo(maxver) } {
        set gDemo(maxver) $ver
    }
}

# Calculates Movement Of Points During Morphing
# This Makes Points Move At A Speed So They All Get To Their
# Destination At The Same Time
proc calculate { i sour dest } {
    global gObjects gDemo

    set sourPoint [lindex $gObjects($sour,points) $i]
    set destPoint [lindex $gObjects($dest,points) $i]
    set x [expr {([lindex $sourPoint 0] - [lindex $destPoint 0]) / $gDemo(steps)}]
    set y [expr {([lindex $sourPoint 1] - [lindex $destPoint 1]) / $gDemo(steps)}]
    set z [expr {([lindex $sourPoint 2] - [lindex $destPoint 2]) / $gDemo(steps)}]
    return [list $x $y $z]
} 

# Resize And Initialize The GL Window
proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h        ; # Reset The Current Viewport
    glMatrixMode GL_PROJECTION  ; # Select The Projection Matrix
    glLoadIdentity              ; # Reset The Projection Matrix

    # Calculate The Aspect Ratio Of The Window
    gluPerspective 45.0 [expr double($w)/double($h)] 0.1 100.0

    glMatrixMode GL_MODELVIEW   ; # Select The Modelview Matrix
    glLoadIdentity              ; # Reset The Modelview Matrix
    set ::gDemo(winWidth)  $w
    set ::gDemo(winHeight) $h
}

# All Setup For OpenGL Goes Here
proc CreateCallback { toglwin } {
    global gObjects gDemo

    glBlendFunc GL_SRC_ALPHA GL_ONE        ; # Set The Blending Function For Translucency
    glClearColor 0.0 0.0 0.0 0.0           ; # This Will Clear The Background Color To Black
    glClearDepth 1.0                       ; # Enables Clearing Of The Depth Buffer
    glDepthFunc GL_LESS                    ; # The Type Of Depth Test To Do
    glEnable GL_DEPTH_TEST                 ; # Enables Depth Testing
    glShadeModel GL_SMOOTH                 ; # Enables Smooth Color Shading
    glHint GL_PERSPECTIVE_CORRECTION_HINT GL_NICEST ; # Really Nice Perspective Calculations

    set gDemo(maxver) 0                    ; # Sets Max Vertices To 0 By Default
    objload "Sphere.txt" 1                 ; # Load First Object Into List gObjects(1,points)
    objload "Torus.txt"  2                 ; # Load Second Object Into List gObjects(2,points)
    objload "Tube.txt"   3                 ; # Load Third Object Into List gObjects(3,points)

    # Fourth Object Gets Random Values
    for { set i 0 } { $i < 486 } { incr i } {
        set x [expr (int(14000*rand())/1000)-7]     ; # Random Float Value From -7 to 7
        set y [expr (int(14000*rand())/1000)-7]     ; # Random Float Value From -7 to 7
        set z [expr (int(14000*rand())/1000)-7]     ; # Random Float Value From -7 to 7
        lappend gObjects(4,points) [list $x $y $z]
    }

    # Load sphere.txt Object Into Helper (Used As Starting Point)
    objload "Sphere.txt" "helper" 

    # Source & Destination Are Set To Equal First Object
    set gDemo(sour) 1
    set gDemo(dest) 1
}

proc Animate {} {
    .fr.toglwin postredisplay
    set ::animateId [tcl3dAfterIdle Animate]
}

proc StartAnimation {} {
    if { ! [info exists ::animateId] } {
        Animate
    }
}

proc StopAnimation {} {
    if { [info exists ::animateId] } {
        after cancel $::animateId 
        unset ::animateId
    }
}

# Here's Where We Do All The Drawing
proc DisplayCallback { toglwin } {
    global gObjects gDemo

    # Clear Screen And Depth Buffer
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]


    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glLoadIdentity
    glTranslatef $gDemo(cx) $gDemo(cy) $gDemo(cz)
    glRotatef $gDemo(xrot) 1 0 0
    glRotatef $gDemo(yrot) 0 1 0
    glRotatef $gDemo(zrot) 0 0 1

    if { [info exists ::animateId] } {
        # Increase xrot,yrot & zrot by xspeed, yspeed & zspeed
        set gDemo(xrot) [expr {$gDemo(xrot) + $gDemo(xspeed)}]
        set gDemo(yrot) [expr {$gDemo(yrot) + $gDemo(yspeed)}]
        set gDemo(zrot) [expr {$gDemo(zrot) + $gDemo(zspeed)}]
    }

    glBegin GL_POINTS
    for { set i 0 } { $i < $gObjects(1,verts) } { incr i } {
        # Loop Through All The Verts Of morph1 (All Objects Have
        # The Same Amount Of Verts For Simplicity, Could Use maxver Also)

        # If morph Is True Calculate Movement Otherwise Movement=0
        if { $gDemo(morph) } {
            set q [calculate $i $gDemo(sour) $gDemo(dest)]
        } else {
            set q [list 0.0 0.0 0.0]
        }

        set helpTmp [lindex $gObjects(helper,points) $i]
        set tx [expr {[lindex $helpTmp 0] - [lindex $q 0]}]
        set ty [expr {[lindex $helpTmp 1] - [lindex $q 1]}]
        set tz [expr {[lindex $helpTmp 2] - [lindex $q 2]}]
        lset gObjects(helper,points) $i [list $tx $ty $tz]

        glColor3f 0 1 1              ; # Set Color To A Bright Shade Of Off Blue
        glVertex3f $tx $ty $tz       ; # Draw A Point At The Current Temp Values (Vertex)
        glColor3f 0 0.5 1            ; # Darken Color A Bit

        # Calculate Two Positions Ahead
        set tx [expr {$tx - 2* [lindex $q 0]}]
        set ty [expr {$ty - 2* [lindex $q 1]}]
        set ty [expr {$ty - 2* [lindex $q 1]}]
        glVertex3f $tx $ty $tz       ; # Draw A Second Point At The Newly Calculate Position
        glColor3f 0 0 1              ; # Set Color To A Very Dark Blue

        # Calculate Two More Positions Ahead
        set tx [expr {$tx - 2* [lindex $q 0]}]
        set ty [expr {$ty - 2* [lindex $q 1]}]
        set ty [expr {$ty - 2* [lindex $q 1]}]
        glVertex3f $tx $ty $tz       ; # Draw A Third Point At The Second New Position
    }
    glEnd

    # If We're Morphing And We Haven't Gone Through All 200 Steps Increase Our Step Counter
    # Otherwise Set Morphing To False, Make Source=Destination And Set The Step Counter Back To Zero.
    if { $gDemo(morph) && $gDemo(step) <= $gDemo(steps) } {
        incr gDemo(step)
    } else {
        set gDemo(morph) false
        set gDemo(sour) $gDemo(dest)
        set gDemo(step) 0
    }
    $toglwin swapbuffers
}

proc Cleanup {} {
    global gObjects gDemo

    catch { unset gObjects }
    catch { unset gDemo }
}

# Put all exit related code here.
proc ExitProg {} {
    exit
}

# Create the OpenGL window and some Tk helper widgets.
proc CreateWindow {} {
    frame .fr
    pack .fr -expand 1 -fill both
    # Create Our OpenGL Window
    togl .fr.toglwin -width $::gDemo(winWidth) -height $::gDemo(winHeight) \
                     -double true -depth true \
                     -createcommand CreateCallback \
                     -reshapecommand ReshapeCallback \
                     -displaycommand DisplayCallback
    listbox .fr.usage -font $::gDemo(listFont) -height 11
    label   .fr.info
    grid .fr.toglwin -row 0 -column 0 -sticky news
    grid .fr.usage   -row 1 -column 0 -sticky news
    grid .fr.info    -row 2 -column 0 -sticky news
    grid rowconfigure .fr 0 -weight 1
    grid columnconfigure .fr 0 -weight 1
    wm title . "Tcl3D demo: Piotr Cieslak & NeHe's Morphing Points Tutorial (Lesson 25)"

    # Watch For ESC Key And Quit Messages
    wm protocol . WM_DELETE_WINDOW "ExitProg"
    bind . <Key-Escape> "ExitProg"
    bind . <Key-F1>     "ToggleWindowMode"
    bind . <Key-Down>   "SetXSpeed  0.01"
    bind . <Key-Up>     "SetXSpeed -0.01"
    bind . <Key-Right>  "SetYSpeed  0.01"
    bind . <Key-Left>   "SetYSpeed -0.01"
    bind . <Key-Prior>  "SetZSpeed  0.01"
    bind . <Key-Next>   "SetZSpeed -0.01"
    bind . <Key-a>      "SetXPos   -0.1"
    bind . <Key-d>      "SetXPos    0.1"
    bind . <Key-s>      "SetYPos   -0.1"
    bind . <Key-w>      "SetYPos    0.1"
    bind . <Key-q>      "SetZPos   -0.1"
    bind . <Key-z>      "SetZPos    0.1"
    bind . <Key-1>      "SwitchMorph 1"
    bind . <Key-2>      "SwitchMorph 2"
    bind . <Key-3>      "SwitchMorph 3"
    bind . <Key-4>      "SwitchMorph 4"
    bind . <Key-r>      "Reset"

    bind .fr.toglwin <1> "StartAnimation"
    bind .fr.toglwin <2> "StopAnimation"
    bind .fr.toglwin <3> "StopAnimation"
    bind .fr.toglwin <Control-Button-1> "StopAnimation"

    .fr.usage insert end "Key-Escape     Exit"
    .fr.usage insert end "Key-F1         Toggle window mode"
    .fr.usage insert end "Key-Up|Down    Decrease|Increase x speed"
    .fr.usage insert end "Key-Left|Right Decrease|Increase y speed"
    .fr.usage insert end "Key-Next|Prior Decrease|Increase z speed"
    .fr.usage insert end "Key-a|d        Decrease|Increase x position"
    .fr.usage insert end "Key-s|w        Decrease|Increase y position"
    .fr.usage insert end "Key-q|z        Decrease|Increase z position"
    .fr.usage insert end "Key-1|2|3|4    Toggle morphs"
    .fr.usage insert end "Key-r          Reset position and rotation"
    .fr.usage insert end "Mouse-L|MR     Start|Stop animation"

    .fr.usage configure -state disabled
}

CreateWindow
PrintInfo [tcl3dOglGetInfoString]
if { [file tail [info script]] eq [file tail $::argv0] } {
    # If started directly from tclsh or wish, then start animation.
    update
    StartAnimation
}
