# Lesson20.tcl
#
# NeHe's Masking Tutorial
#
# This Code Was Created By Jeff Molofee 2000
# And Modified By Giuseppe D'Agata (waveform@tiscalinet.it)
# If You've Found This Code Useful, Please Let Me Know.
# Visit My Site At nehe.gamedev.net
#
# Modified for Tcl3D by Paul Obermeier 2006/03/14
# See www.tcl3d.org for the Tcl3D extension.

package require Img
package require tcl3d

# Font to be used in the Tk listbox.
set gDemo(listFont) {-family {Courier} -size 10}

# Determine the directory of this script.
set gDemo(scriptDir) [file dirname [info script]]

# Window size.
set gDemo(winWidth)  640
set gDemo(winHeight) 480

# Display mode.
set gDemo(fullScreen) 0

set gDemo(masking) 1           ; # Masking On/Off
set gDemo(scene)   0           ; # Which Scene To Draw

set gDemo(texture) [tcl3dVector GLuint 5]      ; # Storage For Five Texture
set gDemo(roll) 0.0                            ; # Rolling Texture

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    ExitProg
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc SetFullScreenMode { win } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win $sw $sh
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+0" $sw $sh]
    wm geometry $win $fmtStr
    wm overrideredirect $win 1
    focus -force $win
}

proc SetWindowMode { win w h } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win 10 10
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+25" $w $h]
    wm geometry $win $fmtStr
    wm overrideredirect $win 0
    focus -force $win
}

# Toggle between windowing and fullscreen mode.
proc ToggleWindowMode {} {
    if { $::gDemo(fullScreen) } {
        SetFullScreenMode .
        set ::gDemo(fullScreen) false
        set ::slowdown 2.0
    } else {
        SetWindowMode . $::gDemo(winWidth) $::gDemo(winHeight)
        set ::gDemo(fullScreen) true
        set ::slowdown 1.0
    }
}

proc LoadGLTextures {} {
    # Load texture images.
    set imgList { "Logo.bmp" "Mask1.bmp" "Image1.bmp" "Mask2.bmp" "Image2.bmp" }

    glGenTextures [llength $imgList] $::gDemo(texture) ; # Create N Textures

    set imgInd 0
    foreach imgName $imgList {
        set texName [file join $::gDemo(scriptDir) "Data" $imgName]
        set retVal [catch {set phImg [image create photo -file $texName]} err1]
        if { $retVal != 0 } {
            error "Error reading image $texName ($err1)"
        } else {
            set w [image width  $phImg]
            set h [image height $phImg]
            set n [tcl3dPhotoChans $phImg]

            set TextureImage [tcl3dVectorFromPhoto $phImg]
            image delete $phImg
        }
        if { $n == 3 } {
            set type $::GL_RGB
        } else {
           set type $::GL_RGBA
        }

        glBindTexture GL_TEXTURE_2D [$::gDemo(texture) get $imgInd]
        glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MAG_FILTER $::GL_LINEAR
        glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MIN_FILTER $::GL_LINEAR
        glTexImage2D GL_TEXTURE_2D 0 $n $w $h 0 $type GL_UNSIGNED_BYTE $TextureImage

        $TextureImage delete       ; # Free The Texture Image Memory
        incr imgInd
    }
}

# Resize And Initialize The GL Window
proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h        ; # Reset The Current Viewport
    glMatrixMode GL_PROJECTION  ; # Select The Projection Matrix
    glLoadIdentity              ; # Reset The Projection Matrix

    # Calculate The Aspect Ratio Of The Window
    gluPerspective 45.0 [expr double($w)/double($h)] 0.1 100.0

    glMatrixMode GL_MODELVIEW   ; # Select The Modelview Matrix
    glLoadIdentity              ; # Reset The Modelview Matrix
    set ::gDemo(winWidth)  $w
    set ::gDemo(winHeight) $h
}

# All Setup For OpenGL Goes Here
proc CreateCallback { toglwin } {
    LoadGLTextures                          ; # Jump To Texture Loading Routine
    glClearColor 0.0 0.0 0.0 0.0            ; # Black Background
    glClearDepth 1.0                        ; # Depth Buffer Setup
    glEnable GL_DEPTH_TEST                  ; # Disable Depth Testing
    glShadeModel GL_SMOOTH                  ; # Enable Smooth Shading
    glEnable GL_TEXTURE_2D                  ; # Enable Texture Mapping
}

proc ToggleMasking {} {
    set ::gDemo(masking) [expr 1 - $::gDemo(masking)]
    .fr.toglwin postredisplay
}

proc ToggleScene {} {
    set ::gDemo(scene) [expr 1 - $::gDemo(scene)]
    .fr.toglwin postredisplay
}

proc Animate {} {
    .fr.toglwin postredisplay
    set ::animateId [tcl3dAfterIdle Animate]
}

proc StartAnimation {} {
    if { ! [info exists ::animateId] } {
        Animate
    }
}

proc StopAnimation {} {
    if { [info exists ::animateId] } {
        after cancel $::animateId 
        unset ::animateId
    }
}

# Here's Where We Do All The Drawing
proc DisplayCallback { toglwin } {
    # Clear Screen And Depth Buffer
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT] 

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glLoadIdentity                      ; # Reset The Current Modelview Matrix

    glTranslatef 0.0 0.0 -2.0           ; # Move Into The Screen 5 Units

    glBindTexture GL_TEXTURE_2D [$::gDemo(texture) get 0]      ; # Select Our Logo Texture
    glBegin GL_QUADS
        glTexCoord2f 0.0 [expr {-1.0*$::gDemo(roll)+0.0}]
        glVertex3f -1.1 -1.1  0.0 ; # Bottom Left
        glTexCoord2f 3.0 [expr {-1.0*$::gDemo(roll)+0.0}]
        glVertex3f  1.1 -1.1  0.0 ; # Bottom Right
        glTexCoord2f 3.0 [expr {-1.0*$::gDemo(roll)+3.0}]
        glVertex3f  1.1  1.1  0.0 ; # Top Right
        glTexCoord2f 0.0 [expr {-1.0*$::gDemo(roll)+3.0}]
        glVertex3f -1.1  1.1  0.0 ; # Top Left
    glEnd

    glEnable GL_BLEND                              ; # Enable Blending
    glDisable GL_DEPTH_TEST                        ; # Disable Depth Testing

    if { $::gDemo(masking) } {
        glBlendFunc GL_DST_COLOR GL_ZERO           ; # Blend Screen Color With Zero (Black)
    }

    if { $::gDemo(scene) } {
        # Are We Drawing The Second Scene?
        glTranslatef 0.0 0.0 -1.0                  ; # Translate Into The Screen One Unit
        glRotatef [expr {$::gDemo(roll)*360}] 0.0 0.0 1.0  ; # Rotate On The Z Axis 360 Degrees.
        if { $::gDemo(masking) } {
            glBindTexture GL_TEXTURE_2D [$::gDemo(texture) get 3]       ; # Select The Second Mask Texture
            glBegin GL_QUADS
                glTexCoord2f 0.0 0.0 ; glVertex3f -1.1 -1.1 0.0 ; # Bottom Left
                glTexCoord2f 1.0 0.0 ; glVertex3f  1.1 -1.1 0.0 ; # Bottom Right
                glTexCoord2f 1.0 1.0 ; glVertex3f  1.1  1.1 0.0 ; # Top Right
                glTexCoord2f 0.0 1.0 ; glVertex3f -1.1  1.1 0.0 ; # Top Left
            glEnd
        }

        glBlendFunc GL_ONE GL_ONE                       ; # Copy Image 2 Color To The Screen
        glBindTexture GL_TEXTURE_2D [$::gDemo(texture) get 4]  ; # Select The Second Image Texture
        glBegin GL_QUADS
            glTexCoord2f 0.0 0.0 ; glVertex3f -1.1 -1.1 0.0 ; # Bottom Left
            glTexCoord2f 1.0 0.0 ; glVertex3f  1.1 -1.1 0.0 ; # Bottom Right
            glTexCoord2f 1.0 1.0 ; glVertex3f  1.1  1.1 0.0 ; # Top Right
            glTexCoord2f 0.0 1.0 ; glVertex3f -1.1  1.1 0.0 ; # Top Left
        glEnd
    } else {
        if { $::gDemo(masking) } {
            glBindTexture GL_TEXTURE_2D [$::gDemo(texture) get 1] ; # Select The First Mask Texture
            glBegin GL_QUADS
                glTexCoord2f [expr {$::gDemo(roll)+0.0}] 0.0 ; glVertex3f -1.1 -1.1 0.0 ; # Bottom Left
                glTexCoord2f [expr {$::gDemo(roll)+4.0}] 0.0 ; glVertex3f  1.1 -1.1 0.0 ; # Bottom Right
                glTexCoord2f [expr {$::gDemo(roll)+4.0}] 4.0 ; glVertex3f  1.1  1.1 0.0 ; # Top Right
                glTexCoord2f [expr {$::gDemo(roll)+0.0}] 4.0 ; glVertex3f -1.1  1.1 0.0 ; # Top Left
            glEnd
        }

        glBlendFunc GL_ONE GL_ONE                       ; # Copy Image 1 Color To The Screen
        glBindTexture GL_TEXTURE_2D [$::gDemo(texture) get 2]  ; # Select The First Image Texture
        glBegin GL_QUADS
            glTexCoord2f [expr {$::gDemo(roll)+0.0}] 0.0 ; glVertex3f -1.1 -1.1 0.0 ; # Bottom Left
            glTexCoord2f [expr {$::gDemo(roll)+4.0}] 0.0 ; glVertex3f  1.1 -1.1 0.0 ; # Bottom Right
            glTexCoord2f [expr {$::gDemo(roll)+4.0}] 4.0 ; glVertex3f  1.1  1.1 0.0 ; # Top Right
            glTexCoord2f [expr {$::gDemo(roll)+0.0}] 4.0 ; glVertex3f -1.1  1.1 0.0 ; # Top Left
        glEnd
    }

    glEnable GL_DEPTH_TEST                              ; # Enable Depth Testing
    glDisable GL_BLEND                                  ; # Disable Blending

    set ::gDemo(roll) [expr {$::gDemo(roll) + 0.002}]   ; # Increase Our Texture Roll Variable
    if { $::gDemo(roll) > 1.0 } {
        set ::gDemo(roll) [expr {$::gDemo(roll) - 1.0}] ; # Subtract 1 From Roll
    }
    $toglwin swapbuffers
}

# Put all exit related code here.
proc ExitProg {} {
    exit
}

# Create the OpenGL window and some Tk helper widgets.
proc CreateWindow {} {
    frame .fr
    pack .fr -expand 1 -fill both
    # Create Our OpenGL Window
    togl .fr.toglwin -width $::gDemo(winWidth) -height $::gDemo(winHeight) \
                     -swapinterval 1 \
                     -double true -depth true \
                     -createcommand CreateCallback \
                     -reshapecommand ReshapeCallback \
                     -displaycommand DisplayCallback 
    listbox .fr.usage -font $::gDemo(listFont) -height 5
    label   .fr.info
    grid .fr.toglwin -row 0 -column 0 -sticky news
    grid .fr.usage   -row 1 -column 0 -sticky news
    grid .fr.info    -row 2 -column 0 -sticky news
    grid rowconfigure .fr 0 -weight 1
    grid columnconfigure .fr 0 -weight 1
    wm title . "Tcl3D demo: NeHe's Masking Tutorial (Lesson 20)"

    # Watch For ESC Key And Quit Messages
    wm protocol . WM_DELETE_WINDOW "ExitProg"
    bind . <Key-Escape> "ExitProg"
    bind . <Key-F1>     "ToggleWindowMode"
    bind . <Key-m>      "ToggleMasking"
    bind . <Key-s>      "ToggleScene"

    bind .fr.toglwin <1> "StartAnimation"
    bind .fr.toglwin <2> "StopAnimation"
    bind .fr.toglwin <3> "StopAnimation"
    bind .fr.toglwin <Control-Button-1> "StopAnimation"

    .fr.usage insert end "Key-Escape  Exit"
    .fr.usage insert end "Key-F1      Toggle window mode"
    .fr.usage insert end "Key-m       Toggle masking"
    .fr.usage insert end "Key-s       Toggle scenes"
    .fr.usage insert end "Mouse-L|MR  Start|Stop animation"

    .fr.usage configure -state disabled
}

CreateWindow
PrintInfo [tcl3dOglGetInfoString]
if { [file tail [info script]] eq [file tail $::argv0] } {
    # If started directly from tclsh or wish, then start animation.
    update
    StartAnimation
}
