# Lesson11.tcl
#
# bosco & NeHe's Waving Texture Tutorial
#
# This Code Was Created By bosco / Jeff Molofee 2000
# A HUGE Thanks To Fredric Echols For Cleaning Up
# And Optimizing This Code, Making It More Flexible!
# If You've Found This Code Useful, Please Let Me Know.
# Visit My Site At nehe.gamedev.net
#
# Modified for Tcl3D by Paul Obermeier 2006/01/25
# See www.tcl3d.org for the Tcl3D extension.

package require Img
package require tcl3d

# Font to be used in the Tk listbox.
set gDemo(listFont) {-family {Courier} -size 10}

# Determine the directory of this script.
set gDemo(scriptDir) [file dirname [info script]]

# Display mode.
set gDemo(fullScreen) false

# Window size.
set gDemo(winWidth)  640
set gDemo(winHeight) 480

set gDemo(wiggleCount) 0      ; # Counter Used To Control How Fast Flag Waves
set gDemo(xrot) 0.0            ; # X Rotation ( NEW )
set gDemo(yrot) 0.0            ; # Y Rotation ( NEW )
set gDemo(zrot) 0.0            ; # Z Rotation ( NEW )

set gDemo(texture) [tcl3dVector GLuint 1] ; # Storage For One Texture ( NEW )

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    ExitProg
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc SetFullScreenMode { win } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win $sw $sh
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+0" $sw $sh]
    wm geometry $win $fmtStr
    wm overrideredirect $win 1
    focus -force $win
}

proc SetWindowMode { win w h } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win 10 10
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+25" $w $h]
    wm geometry $win $fmtStr
    wm overrideredirect $win 0
    focus -force $win
}

# Toggle between windowing and fullscreen mode.
proc ToggleWindowMode {} {
    if { $::gDemo(fullScreen) } {
        SetFullScreenMode .
        set ::gDemo(fullScreen) false
    } else {
        SetWindowMode . $::gDemo(winWidth) $::gDemo(winHeight)
        set ::gDemo(fullScreen) true
    }
}

proc LoadGLTextures {} {
    # Load texture image.
    set texName [file join $::gDemo(scriptDir) "Data" "Tim.bmp"]
    set retVal [catch {set phImg [image create photo -file $texName]} err1]
    if { $retVal != 0 } {
        error "Error reading image $texName ($err1)"
    } else {
        set w [image width  $phImg]
        set h [image height $phImg]
        set n [tcl3dPhotoChans $phImg]
        set TextureImage [tcl3dVectorFromPhoto $phImg]
        image delete $phImg
    }

    glGenTextures 1 $::gDemo(texture)          ; # Create The Texture

    # Typical Texture Generation Using Data From The Bitmap
    glBindTexture GL_TEXTURE_2D [$::gDemo(texture) get 0]
    if { $n == 3 } {
        set type $::GL_RGB
    } else {
       set type $::GL_RGBA
    }
    glTexImage2D GL_TEXTURE_2D 0 $n $w $h 0 $type GL_UNSIGNED_BYTE $TextureImage

    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MIN_FILTER $::GL_LINEAR
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MAG_FILTER $::GL_LINEAR

    $TextureImage delete       ; # Free The Texture Image Memory
}

# Resize And Initialize The GL Window
proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h        ; # Reset The Current Viewport
    glMatrixMode GL_PROJECTION  ; # Select The Projection Matrix
    glLoadIdentity              ; # Reset The Projection Matrix

    # Calculate The Aspect Ratio Of The Window
    gluPerspective 45.0 [expr double($w)/double($h)] 0.1 100.0

    glMatrixMode GL_MODELVIEW   ; # Select The Modelview Matrix
    glLoadIdentity              ; # Reset The Modelview Matrix
    set ::gDemo(winWidth)  $w
    set ::gDemo(winHeight) $h
}

# All Setup For OpenGL Goes Here
proc CreateCallback { toglwin } {
    LoadGLTextures                          ; # Jump To Texture Loading Routine ( NEW )
    glEnable GL_TEXTURE_2D                  ; # Enable Texture Mapping ( NEW )

    glShadeModel GL_SMOOTH                  ; # Enable Smooth Shading
    glClearColor 0.0 0.0 0.0 0.5            ; # Black Background
    glClearDepth 1.0                        ; # Depth Buffer Setup
    glEnable GL_DEPTH_TEST                  ; # Enables Depth Testing
    glDepthFunc GL_LEQUAL                   ; # The Type Of Depth Testing To Do
    glHint GL_PERSPECTIVE_CORRECTION_HINT GL_NICEST ; # Really Nice Perspective Calculations

    glPolygonMode GL_BACK GL_FILL           ; # Back Face Is Solid
    glPolygonMode GL_FRONT GL_LINE          ; # Front Face Is Made Of Lines

    for { set x 0 } { $x < 45 } { incr x } {
        for { set y 0 } { $y < 45 } { incr y } {
            set ::points($x,$y,0) [expr double(($x/5.0)-4.5)]
            set ::points($x,$y,1) [expr double(($y/5.0)-4.5)]
            set ::points($x,$y,2) [expr double(sin(((($x/5.0)*40.0)/360.0)*3.141592654*2.0))]
                }
        }
}

proc Animate {} {
    .fr.toglwin postredisplay
    set ::animateId [tcl3dAfterIdle Animate]
}

proc StartAnimation {} {
    if { ! [info exists ::animateId] } {
        Animate
    }
}

proc StopAnimation {} {
    if { [info exists ::animateId] } {
        after cancel $::animateId 
        unset ::animateId
    }
}

# Here's Where We Do All The Drawing
proc DisplayCallback { toglwin } {
    # Clear Screen And Depth Buffer
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT] 

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glLoadIdentity                      ; # Reset The Current Modelview Matrix
    glTranslatef 0.0 0.0 -12.0
    glRotatef $::gDemo(xrot) 1.0 0.0 0.0
    glRotatef $::gDemo(yrot) 0.0 1.0 0.0
    glRotatef $::gDemo(zrot) 0.0 0.0 1.0

    glBindTexture GL_TEXTURE_2D [$::gDemo(texture) get 0]

    glBegin GL_QUADS
        for { set x 0 } { $x < 44 } { incr x } {
            for { set y 0 } { $y < 44 } { incr y } {
                set x1 [expr {$x + 1}]
                set y1 [expr {$y + 1}]
                set float_x  [expr {double($x)/44.0}]
                set float_y  [expr {double($y)/44.0}]
                set float_xb [expr {double($x1)/44.0}]
                set float_yb [expr {double($y1)/44.0}]

                glTexCoord2f $float_x $float_y
                glVertex3f $::points($x,$y,0) $::points($x,$y,1) $::points($x,$y,2)

                glTexCoord2f $float_x $float_yb
                glVertex3f $::points($x,$y1,0) $::points($x,$y1,1) $::points($x,$y1,2)

                glTexCoord2f $float_xb $float_yb
                glVertex3f $::points($x1,$y1,0) $::points($x1,$y1,1) $::points($x1,$y1,2)

                glTexCoord2f $float_xb $float_y
                glVertex3f $::points($x1,$y,0) $::points($x1,$y,1) $::points($x1,$y,2)
            }
        }
    glEnd

    if { $::gDemo(wiggleCount) == 2 } {
        for { set y 0 } { $y < 45 } { incr y } {
            set hold $::points(0,$y,2)
            for { set x 0 } { $x < 44 } { incr x } {
                set ::points($x,$y,2) $::points([expr {$x+1}],$y,2)
            }
            set ::points(44,$y,2) $hold
        }
        set ::gDemo(wiggleCount) 0
    }

    incr ::gDemo(wiggleCount)
        
    set ::gDemo(xrot) [expr $::gDemo(xrot) + 0.3]
    set ::gDemo(yrot) [expr $::gDemo(yrot) + 0.2]
    set ::gDemo(zrot) [expr $::gDemo(zrot) + 0.4]
    
    $toglwin swapbuffers
}

# Put all exit related code here.
proc ExitProg {} {
    exit
}

# Create the OpenGL window and some Tk helper widgets.
proc CreateWindow {} {
    frame .fr
    pack .fr -expand 1 -fill both
    # Create Our OpenGL Window
    togl .fr.toglwin -width $::gDemo(winWidth) -height $::gDemo(winHeight) \
                     -double true -depth true \
                     -createcommand CreateCallback \
                     -reshapecommand ReshapeCallback \
                     -displaycommand DisplayCallback 
    listbox .fr.usage -font $::gDemo(listFont) -height 3
    label   .fr.info
    grid .fr.toglwin -row 0 -column 0 -sticky news
    grid .fr.usage   -row 1 -column 0 -sticky news
    grid .fr.info    -row 2 -column 0 -sticky news
    grid rowconfigure .fr 0 -weight 1
    grid columnconfigure .fr 0 -weight 1
    wm title . "Tcl3D demo: bosco & NeHe's Waving Texture Tutorial (Lesson 11)"

    # Watch For ESC Key And Quit Messages
    wm protocol . WM_DELETE_WINDOW "ExitProg"
    bind . <Key-Escape> "ExitProg"
    bind . <Key-F1>     "ToggleWindowMode"

    bind .fr.toglwin <1> "StartAnimation"
    bind .fr.toglwin <2> "StopAnimation"
    bind .fr.toglwin <3> "StopAnimation"
    bind .fr.toglwin <Control-Button-1> "StopAnimation"

    .fr.usage insert end "Key-Escape Exit"
    .fr.usage insert end "Key-F1     Toggle window mode"
    .fr.usage insert end "Mouse-L|MR Start|Stop animation"

    .fr.usage configure -state disabled
}

CreateWindow
PrintInfo [tcl3dOglGetInfoString]
if { [file tail [info script]] eq [file tail $::argv0] } {
    # If started directly from tclsh or wish, then start animation.
    update
    StartAnimation
}
