# Lesson09.tcl
#
# NeHe's Animated Blended Textures Tutorial
#
# This Code Was Created By Jeff Molofee 2000
# A HUGE Thanks To Fredric Echols For Cleaning Up
# And Optimizing This Code, Making It More Flexible!
# If You've Found This Code Useful, Please Let Me Know.
# Visit My Site At nehe.gamedev.net
#
# Modified for Tcl3D by Paul Obermeier 2006/01/25
# See www.tcl3d.org for the Tcl3D extension.

package require Img
package require tcl3d

# Font to be used in the Tk listbox.
set gDemo(listFont) {-family {Courier} -size 10}

# Determine the directory of this script.
set gDemo(scriptDir) [file dirname [info script]]

# Display mode.
set gDemo(fullScreen) false

# Window size.
set gDemo(winWidth)  640
set gDemo(winHeight) 480

set gDemo(twinkle) false       ; # Twinkling Stars

set gDemo(num)   50            ; # Number Of Stars To Draw
set gDemo(zoom) -15.0          ; # Distance Away From Stars
set gDemo(tilt)  90.0          ; # Tilt The View
set gDemo(spin)  0.0           ; # Spin Stars

set gDemo(texture) [tcl3dVector GLuint 1] ; # Storage For One Texture

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    ExitProg
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc SetFullScreenMode { win } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win $sw $sh
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+0" $sw $sh]
    wm geometry $win $fmtStr
    wm overrideredirect $win 1
    focus -force $win
}

proc SetWindowMode { win w h } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win 10 10
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+25" $w $h]
    wm geometry $win $fmtStr
    wm overrideredirect $win 0
    focus -force $win
}

# Toggle between windowing and fullscreen mode.
proc ToggleWindowMode {} {
    if { $::gDemo(fullScreen) } {
        SetFullScreenMode .
        set ::gDemo(fullScreen) false
    } else {
        SetWindowMode . $::gDemo(winWidth) $::gDemo(winHeight)
        set ::gDemo(fullScreen) true
    }
}

# Toggle twinkling.
proc ToggleTwinkle {} {
    set ::gDemo(twinkle) [expr ! $::gDemo(twinkle)]
    .fr.toglwin postredisplay
}

# Increment tilt angle.
proc IncrTilt { val } {
    set ::gDemo(tilt) [expr {$::gDemo(tilt) + $val}]
    .fr.toglwin postredisplay
}

# Increment zoom factor.
proc IncrZoom { val } {
    set ::gDemo(zoom) [expr {$::gDemo(zoom) + $val}]
    .fr.toglwin postredisplay
}

proc LoadGLTextures {} {
    # Load texture image.
    set texName [file join $::gDemo(scriptDir) "Data" "Star.bmp"]
    set retVal [catch {set phImg [image create photo -file $texName]} err1]
    if { $retVal != 0 } {
        error "Error reading image $texName ($err1)"
    } else {
        set w [image width  $phImg]
        set h [image height $phImg]
        set n [tcl3dPhotoChans $phImg]
        set TextureImage [tcl3dVectorFromPhoto $phImg]
        image delete $phImg
    }

    glGenTextures 1 $::gDemo(texture)          ; # Create One Texture

    if { $n == 3 } {
        set type $::GL_RGB
    } else {
       set type $::GL_RGBA
    }
    # Create Linear Filtered Texture
    glBindTexture GL_TEXTURE_2D [$::gDemo(texture) get 0]
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MIN_FILTER $::GL_LINEAR
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MAG_FILTER $::GL_LINEAR
    glTexImage2D GL_TEXTURE_2D 0 $n $w $h 0 $type GL_UNSIGNED_BYTE $TextureImage

    $TextureImage delete       ; # Free The Texture Image Memory
}

# Resize And Initialize The GL Window
proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h        ; # Reset The Current Viewport
    glMatrixMode GL_PROJECTION  ; # Select The Projection Matrix
    glLoadIdentity              ; # Reset The Projection Matrix

    # Calculate The Aspect Ratio Of The Window
    gluPerspective 45.0 [expr double($w)/double($h)] 0.1 100.0

    glMatrixMode GL_MODELVIEW   ; # Select The Modelview Matrix
    glLoadIdentity              ; # Reset The Modelview Matrix
    set ::gDemo(winWidth)  $w
    set ::gDemo(winHeight) $h
}

# All Setup For OpenGL Goes Here
proc CreateCallback { toglwin } {
    LoadGLTextures                          ; # Jump To Texture Loading Routine
    glEnable GL_TEXTURE_2D                  ; # Enable Texture Mapping

    glShadeModel GL_SMOOTH                  ; # Enable Smooth Shading
    glClearColor 0.0 0.0 0.0 0.5            ; # Black Background
    glClearDepth 1.0                        ; # Depth Buffer Setup
    glHint GL_PERSPECTIVE_CORRECTION_HINT GL_NICEST ; # Really Nice Perspective Calculations
    glBlendFunc GL_SRC_ALPHA GL_ONE         ; # Set The Blending Function For Translucency
    glEnable GL_BLEND

    for { set loop 0 } { $loop < $::gDemo(num) } { incr loop } {
        set ::star($loop,angle) 0.0
        set ::star($loop,dist) [expr (double($loop)/double($::gDemo(num)))*5.0]
        set ::star($loop,r) [expr int (rand() * 256)]
        set ::star($loop,g) [expr int (rand() * 256)]
        set ::star($loop,b) [expr int (rand() * 256)]
    }
}

proc Animate {} {
    .fr.toglwin postredisplay
    set ::animateId [tcl3dAfterIdle Animate]
}

proc StartAnimation {} {
    if { ! [info exists ::animateId] } {
        Animate
    }
}

proc StopAnimation {} {
    if { [info exists ::animateId] } {
        after cancel $::animateId 
        unset ::animateId
    }
}

# Here's Where We Do All The Drawing
proc DisplayCallback { toglwin } {
    # Clear Screen And Depth Buffer
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glBindTexture GL_TEXTURE_2D [$::gDemo(texture) get 0]  ; # Select Our Texture

    # Loop Through All The Stars
    for { set loop 0 } { $loop < $::gDemo(num) } { incr loop } {
        glLoadIdentity                              ; # Reset The View Before We Draw Each Star
        glTranslatef 0.0 0.0 $::gDemo(zoom)         ; # Zoom Into The Screen (Using The Value In 'zoom')
        glRotatef $::gDemo(tilt) 1.0 0.0 0.0        ; # Tilt The View (Using The Value In 'tilt')
        glRotatef $::star($loop,angle) 0.0 1.0 0.0  ; # Rotate To The Current Stars Angle
        glTranslatef $::star($loop,dist) 0.0 0.0    ; # Move Forward On The X Plane
        glRotatef [expr {-1.0 * $::star($loop,angle)}] 0.0 1.0 0.0 ; # Cancel The Current Stars Angle
        glRotatef [expr {-1.0 * $::gDemo(tilt)}] 1.0 0.0 0.0       ; # Cancel The Screen Tilt
        
        if { $::gDemo(twinkle) } {
            set ind [expr {$::gDemo(num) - $loop -1}]
            glColor4ub $::star($ind,r) $::star($ind,g) $::star($ind,b) 255
            glBegin GL_QUADS
                glTexCoord2f 0.0 0.0 ; glVertex3f -1.0 -1.0 0.0
                glTexCoord2f 1.0 0.0 ; glVertex3f  1.0 -1.0 0.0
                glTexCoord2f 1.0 1.0 ; glVertex3f  1.0  1.0 0.0
                glTexCoord2f 0.0 1.0 ; glVertex3f -1.0  1.0 0.0
            glEnd
        }

        glRotatef $::gDemo(spin) 0.0 0.0 1.0
        glColor4ub $::star($loop,r) $::star($loop,g) $::star($loop,b) 255
        glBegin GL_QUADS
            glTexCoord2f 0.0  0.0 ; glVertex3f -1.0 -1.0  0.0
            glTexCoord2f 1.0  0.0 ; glVertex3f  1.0 -1.0  0.0
            glTexCoord2f 1.0  1.0 ; glVertex3f  1.0  1.0  0.0
            glTexCoord2f 0.0  1.0 ; glVertex3f -1.0  1.0  0.0
        glEnd

        if { [info exists ::animateId] } {
            set ::gDemo(spin) [expr {$::gDemo(spin) + 0.01}]
            set ::star($loop,angle) [expr {$::star($loop,angle) + double($loop)/$::gDemo(num)}]
            set ::star($loop,dist)  [expr {$::star($loop,dist) - 0.01}]
            if { $::star($loop,dist) < 0.0 } {
                set ::star($loop,dist) [expr {$::star($loop,dist) + 5.0}]
                set ::star($loop,r) [expr {int (rand() * 256)}] 
                set ::star($loop,g) [expr {int (rand() * 256)}]
                set ::star($loop,b) [expr {int (rand() * 256)}]
            }
        }
    }
    $toglwin swapbuffers
}

proc Cleanup {} {
    glDeleteTextures 1 [$::gDemo(texture) get 0]
    $::gDemo(texture) delete
    uplevel #0 unset gDemo
}

# Put all exit related code here.
proc ExitProg {} {
    exit
}

# Create the OpenGL window and some Tk helper widgets.
proc CreateWindow {} {
    frame .fr
    pack .fr -expand 1 -fill both
    # Create Our OpenGL Window
    togl .fr.toglwin -width $::gDemo(winWidth) -height $::gDemo(winHeight) \
                     -double true -depth true \
                     -swapinterval 1 \
                     -createcommand CreateCallback \
                     -reshapecommand ReshapeCallback \
                     -displaycommand DisplayCallback 
    listbox .fr.usage -font $::gDemo(listFont) -height 6
    label   .fr.info
    grid .fr.toglwin -row 0 -column 0 -sticky news
    grid .fr.usage   -row 1 -column 0 -sticky news
    grid .fr.info    -row 2 -column 0 -sticky news
    grid rowconfigure .fr 0 -weight 1
    grid columnconfigure .fr 0 -weight 1
    wm title . "Tcl3D demo: NeHe's Animated Blended Textures Tutorial (Lesson 9)"

    # Watch For ESC Key And Quit Messages
    wm protocol . WM_DELETE_WINDOW "ExitProg"
    bind . <Key-Escape> "ExitProg"
    bind . <Key-F1>     "ToggleWindowMode"
    bind . <Key-t>      "ToggleTwinkle"
    bind . <Key-Up>     "IncrTilt -1.0"
    bind . <Key-Down>   "IncrTilt  1.0"
    bind . <Key-d>      "IncrZoom  0.2"
    bind . <Key-i>      "IncrZoom -0.2"

    bind .fr.toglwin <1> "StartAnimation"
    bind .fr.toglwin <2> "StopAnimation"
    bind .fr.toglwin <3> "StopAnimation"
    bind .fr.toglwin <Control-Button-1> "StopAnimation"

    .fr.usage insert end "Key-Escape  Exit"
    .fr.usage insert end "Key-F1      Toggle window mode"
    .fr.usage insert end "Key-t       Toggle twinkle"
    .fr.usage insert end "Key-Up|Down Decrease|Increase tilt"
    .fr.usage insert end "Key-d|i     Decrease|Increase distance"
    .fr.usage insert end "Mouse-L|MR  Start|Stop animation"

    .fr.usage configure -state disabled
}

CreateWindow
PrintInfo [tcl3dOglGetInfoString]
if { [file tail [info script]] eq [file tail $::argv0] } {
    # If started directly from tclsh or wish, then start animation.
    update
    StartAnimation
}
