# Lesson08.tcl
#
# Tom Stanis & NeHe's Blending Tutorial
#
# This Code Was Created By Tom Stanis / Jeff Molofee 2000
# A HUGE Thanks To Fredric Echols For Cleaning Up
# And Optimizing This Code, Making It More Flexible!
# If You've Found This Code Useful, Please Let Me Know.
# Visit My Site At nehe.gamedev.net
#
# Modified for Tcl3D by Paul Obermeier 2006/01/25
# See www.tcl3d.org for the Tcl3D extension.

package require Img
package require tcl3d

# Font to be used in the Tk listbox.
set gDemo(listFont) {-family {Courier} -size 10}

# Determine the directory of this script.
set gDemo(scriptDir) [file dirname [info script]]

# Display mode.
set gDemo(fullScreen) false

# Window size.
set gDemo(winWidth)  640
set gDemo(winHeight) 480

set gDemo(light) false ; # Lighting ON/OFF
set gDemo(blend) false ; # Blending ON/OFF (NEW)

set gDemo(xrot)   0.0  ; # X Rotation
set gDemo(yrot)   0.0  ; # Y Rotation
set gDemo(xspeed) 0.5  ; # X Rotation Speed
set gDemo(yspeed) 0.5  ; # Y Rotation Speed
set gDemo(z)     -5.0  ; # Depth Into The Screen

set gDemo(lightAmbient) { 0.5 0.5 0.5 1.0 }
set gDemo(lightDiffuse) { 1.0 1.0 1.0 1.0 }
set gDemo(lightPos)     { 0.0 0.0 2.0 1.0 }

set gDemo(filter) -1                       ; # Which Filter To Use
set gDemo(filterList) [list "Nearest" "Linear" "MipMapped"]

set gDemo(texture) [tcl3dVector GLuint 3] ; # Storage For 3 Textures

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    ExitProg
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc SetFullScreenMode { win } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win $sw $sh
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+0" $sw $sh]
    wm geometry $win $fmtStr
    wm overrideredirect $win 1
    focus -force $win
}

proc SetWindowMode { win w h } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win 10 10
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+25" $w $h]
    wm geometry $win $fmtStr
    wm overrideredirect $win 0
    focus -force $win
}

# Toggle between windowing and fullscreen mode.
proc ToggleWindowMode {} {
    if { $::gDemo(fullScreen) } {
        SetFullScreenMode .
        set ::gDemo(fullScreen) false
    } else {
        SetWindowMode . $::gDemo(winWidth) $::gDemo(winHeight)
        set ::gDemo(fullScreen) true
    }
}

# Toggle lighting on or off.
proc ToggleLighting {} {
    set ::gDemo(light) [expr ! $::gDemo(light)]
    if { $::gDemo(light) } {
        glEnable GL_LIGHTING
    } else {
        glDisable GL_LIGHTING
    }
    .fr.toglwin postredisplay
}

# Toggle blending on or off. (NEW)
proc ToggleBlending {} {
    set ::gDemo(blend) [expr ! $::gDemo(blend)]
    if { $::gDemo(blend) } {
        glEnable GL_BLEND           ; # Turn Blending On
        glDisable GL_DEPTH_FUNC     ; # Turn Depth Testing Off
    } else {
        glDisable GL_BLEND          ; # Turn Blending Off
        glEnable GL_DEPTH_FUNC      ; # Turn Depth Testing On
    }
    .fr.toglwin postredisplay
}

# Toggle between the different texture filters.
proc ToggleFilter {} {
    incr ::gDemo(filter)
    if { $::gDemo(filter) > 2 } {
        set ::gDemo(filter) 0
    }
    set filterName [lindex $::gDemo(filterList) $::gDemo(filter)]
    PrintInfo [format "Filter %s\n%s" $filterName $::glInfo]
    .fr.toglwin postredisplay
}

# Set the rotation speed in X.
proc SetXSpeed { val } {
    set ::gDemo(xspeed) [expr $::gDemo(xspeed) + $val]
}

# Set the rotation speed in Y.
proc SetYSpeed { val } {
    set ::gDemo(yspeed) [expr $::gDemo(yspeed) + $val]
}

# Set the depth.
proc SetDepth { val } {
    set ::gDemo(z) [expr $::gDemo(z) + $val]
    .fr.toglwin postredisplay
}

proc LoadGLTextures {} {
    # Load texture image.
    set texName [file join $::gDemo(scriptDir) "Data" "Glass.bmp"]
    set retVal [catch {set phImg [image create photo -file $texName]} err1]
    if { $retVal != 0 } {
        error "Error reading image $texName ($err1)"
    } else {
        set w [image width  $phImg]
        set h [image height $phImg]
        set n [tcl3dPhotoChans $phImg]
        set TextureImage [tcl3dVectorFromPhoto $phImg]
        image delete $phImg
    }
    if { $n == 3 } {
        set type $::GL_RGB
    } else {
       set type $::GL_RGBA
    }

    glGenTextures 3 $::gDemo(texture)          ; # Create Three Textures

    # Create Nearest Filtered Texture
    glBindTexture GL_TEXTURE_2D [$::gDemo(texture) get 0]
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MAG_FILTER $::GL_NEAREST
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MIN_FILTER $::GL_NEAREST
    glTexImage2D GL_TEXTURE_2D 0 $n $w $h 0 $type GL_UNSIGNED_BYTE $TextureImage

    # Create Linear Filtered Texture
    glBindTexture GL_TEXTURE_2D [$::gDemo(texture) get 1]
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MAG_FILTER $::GL_LINEAR
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MIN_FILTER $::GL_LINEAR
    glTexImage2D GL_TEXTURE_2D 0 $n $w $h 0 $type GL_UNSIGNED_BYTE $TextureImage

    # Create MipMapped Texture
    glBindTexture GL_TEXTURE_2D [$::gDemo(texture) get 2]
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MAG_FILTER $::GL_LINEAR
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MIN_FILTER $::GL_LINEAR_MIPMAP_NEAREST
    glTexImage2D GL_TEXTURE_2D 0 $n $w $h 0 $type GL_UNSIGNED_BYTE $TextureImage
    gluBuild2DMipmaps GL_TEXTURE_2D $n $w $h $type GL_UNSIGNED_BYTE $TextureImage

    $TextureImage delete       ; # Free The Texture Image Memory
}

# Resize And Initialize The GL Window
proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h        ; # Reset The Current Viewport
    glMatrixMode GL_PROJECTION  ; # Select The Projection Matrix
    glLoadIdentity              ; # Reset The Projection Matrix

    # Calculate The Aspect Ratio Of The Window
    gluPerspective 45.0 [expr double($w)/double($h)] 0.1 100.0

    glMatrixMode GL_MODELVIEW   ; # Select The Modelview Matrix
    glLoadIdentity              ; # Reset The Modelview Matrix
    set ::gDemo(winWidth)  $w
    set ::gDemo(winHeight) $h
}

# All Setup For OpenGL Goes Here
proc CreateCallback { toglwin } {
    LoadGLTextures                          ; # Jump To Texture Loading Routine
    glEnable GL_TEXTURE_2D                  ; # Enable Texture Mapping

    glShadeModel GL_SMOOTH                  ; # Enable Smooth Shading
    glClearColor 0.0 0.0 0.0 0.5            ; # Black Background
    glClearDepth 1.0                        ; # Depth Buffer Setup
    glEnable GL_DEPTH_TEST                  ; # Enables Depth Testing
    glDepthFunc GL_LEQUAL                   ; # The Type Of Depth Testing To Do
    glHint GL_PERSPECTIVE_CORRECTION_HINT GL_NICEST ; # Really Nice Perspective Calculations

    glLightfv GL_LIGHT1 GL_AMBIENT  $::gDemo(lightAmbient)     ; # Setup The Ambient Light
    glLightfv GL_LIGHT1 GL_DIFFUSE  $::gDemo(lightDiffuse)     ; # Setup The Diffuse Light
    glLightfv GL_LIGHT1 GL_POSITION $::gDemo(lightPos)    ; # Position The Light
    glEnable GL_LIGHT1                                  ; # Enable Light One

    glColor4f 1.0 1.0 1.0 0.5                           ; # Full Brightness. 50% Alpha
    glBlendFunc GL_SRC_ALPHA GL_ONE                     ; # Set The Blending Function For Translucency
}

proc Animate {} {
    .fr.toglwin postredisplay
    set ::animateId [tcl3dAfterIdle Animate]
}

proc StartAnimation {} {
    if { ! [info exists ::animateId] } {
        Animate
    }
}

proc StopAnimation {} {
    if { [info exists ::animateId] } {
        after cancel $::animateId 
        unset ::animateId
    }
}

# Here's Where We Do All The Drawing
proc DisplayCallback { toglwin } {
    # Clear Screen And Depth Buffer
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT] 

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glLoadIdentity                      ; # Reset The Current Modelview Matrix
    glTranslatef 0.0 0.0 $::gDemo(z)
    glRotatef $::gDemo(xrot) 1.0 0.0 0.0
    glRotatef $::gDemo(yrot) 0.0 1.0 0.0

    glBindTexture GL_TEXTURE_2D [$::gDemo(texture) get $::gDemo(filter)]

    glBegin GL_QUADS
        # Front Face
        glNormal3f 0.0 0.0 1.0
        glTexCoord2f 0.0 0.0 ; glVertex3f -1.0 -1.0  1.0 ;
        glTexCoord2f 1.0 0.0 ; glVertex3f  1.0 -1.0  1.0 ;
        glTexCoord2f 1.0 1.0 ; glVertex3f  1.0  1.0  1.0 ;
        glTexCoord2f 0.0 1.0 ; glVertex3f -1.0  1.0  1.0 ;
        # Back Face
        glNormal3f 0.0 0.0 -1.0
        glTexCoord2f 1.0 0.0 ; glVertex3f -1.0 -1.0 -1.0 ;
        glTexCoord2f 1.0 1.0 ; glVertex3f -1.0  1.0 -1.0 ;
        glTexCoord2f 0.0 1.0 ; glVertex3f  1.0  1.0 -1.0 ;
        glTexCoord2f 0.0 0.0 ; glVertex3f  1.0 -1.0 -1.0 ;
        # Top Face
        glNormal3f 0.0 1.0 0.0
        glTexCoord2f 0.0 1.0 ; glVertex3f -1.0  1.0 -1.0 ;
        glTexCoord2f 0.0 0.0 ; glVertex3f -1.0  1.0  1.0 ;
        glTexCoord2f 1.0 0.0 ; glVertex3f  1.0  1.0  1.0 ;
        glTexCoord2f 1.0 1.0 ; glVertex3f  1.0  1.0 -1.0 ;
        # Bottom Face
        glNormal3f 0.0 -1.0 0.0
        glTexCoord2f 1.0 1.0 ; glVertex3f -1.0 -1.0 -1.0 ;
        glTexCoord2f 0.0 1.0 ; glVertex3f  1.0 -1.0 -1.0 ;
        glTexCoord2f 0.0 0.0 ; glVertex3f  1.0 -1.0  1.0 ;
        glTexCoord2f 1.0 0.0 ; glVertex3f -1.0 -1.0  1.0 ;
        # Right face
        glNormal3f 1.0 0.0 0.0
        glTexCoord2f 1.0 0.0 ; glVertex3f  1.0 -1.0 -1.0 ;
        glTexCoord2f 1.0 1.0 ; glVertex3f  1.0  1.0 -1.0 ;
        glTexCoord2f 0.0 1.0 ; glVertex3f  1.0  1.0  1.0 ;
        glTexCoord2f 0.0 0.0 ; glVertex3f  1.0 -1.0  1.0 ;
        # Left Face
        glNormal3f -1.0 0.0 0.0
        glTexCoord2f 0.0 0.0 ; glVertex3f -1.0 -1.0 -1.0 ;
        glTexCoord2f 1.0 0.0 ; glVertex3f -1.0 -1.0  1.0 ;
        glTexCoord2f 1.0 1.0 ; glVertex3f -1.0  1.0  1.0 ;
        glTexCoord2f 0.0 1.0 ; glVertex3f -1.0  1.0 -1.0 ;
    glEnd

    if { [info exists ::animateId] } {
        set ::gDemo(xrot) [expr $::gDemo(xrot) + $::gDemo(xspeed)]
        set ::gDemo(yrot) [expr $::gDemo(yrot) + $::gDemo(yspeed)]
    }
    
    $toglwin swapbuffers
}

proc Cleanup {} {
    glDeleteTextures 3 [$::gDemo(texture) get 0]
    $::gDemo(texture) delete
    uplevel #0 unset gDemo
}

# Put all exit related code here.
proc ExitProg {} {
    exit
}

# Create the OpenGL window and some Tk helper widgets.
proc CreateWindow {} {
    frame .fr
    pack .fr -expand 1 -fill both
    # Create Our OpenGL Window
    togl .fr.toglwin -width $::gDemo(winWidth) -height $::gDemo(winHeight) \
                     -swapinterval 1 \
                     -double true -depth true \
                     -createcommand CreateCallback \
                     -reshapecommand ReshapeCallback \
                     -displaycommand DisplayCallback
    listbox .fr.usage -font $::gDemo(listFont) -height 9
    label   .fr.info
    grid .fr.toglwin -row 0 -column 0 -sticky news
    grid .fr.usage   -row 1 -column 0 -sticky news
    grid .fr.info    -row 2 -column 0 -sticky news
    grid rowconfigure .fr 0 -weight 1
    grid columnconfigure .fr 0 -weight 1
    wm title . "Tcl3D demo: Tom Stanis & NeHe's Blending Tutorial (Lesson 8)"

    # Watch For ESC Key And Quit Messages
    wm protocol . WM_DELETE_WINDOW "ExitProg"
    bind . <Key-Escape> "ExitProg"
    bind . <Key-F1>     "ToggleWindowMode"
    bind . <Key-l>      "ToggleLighting"
    bind . <Key-f>      "ToggleFilter"
    bind . <Key-b>      "ToggleBlending"
    bind . <Key-Up>     "SetXSpeed -0.01"
    bind . <Key-Down>   "SetXSpeed  0.01"
    bind . <Key-Left>   "SetYSpeed -0.01"
    bind . <Key-Right>  "SetYSpeed  0.01"
    bind . <Key-d>      "SetDepth  0.05"
    bind . <Key-i>      "SetDepth -0.05"

    bind .fr.toglwin <1> "StartAnimation"
    bind .fr.toglwin <2> "StopAnimation"
    bind .fr.toglwin <3> "StopAnimation"
    bind .fr.toglwin <Control-Button-1> "StopAnimation"

    .fr.usage insert end "Key-Escape     Exit"
    .fr.usage insert end "Key-F1         Toggle window mode"
    .fr.usage insert end "Key-l          Toggle lighting"
    .fr.usage insert end "Key-f          Toggle filter"
    .fr.usage insert end "Key-b          Toggle blending"
    .fr.usage insert end "Key-Up|Down    Decrease|Increase x rotation speed"
    .fr.usage insert end "Key-Left|Right Decrease|Increase y rotation speed"
    .fr.usage insert end "Key-d|i        Decrease|Increase distance"
    .fr.usage insert end "Mouse-L|MR     Start|Stop animation"

    .fr.usage configure -state disabled
}

CreateWindow
set glInfo [tcl3dOglGetInfoString]
ToggleFilter
if { [file tail [info script]] eq [file tail $::argv0] } {
    # If started directly from tclsh or wish, then start animation.
    update
    StartAnimation
}
