# Lesson05.tcl
#
# NeHe's Solid Object Tutorial
#
# This Code Was Created By Jeff Molofee 2000
# A HUGE Thanks To Fredric Echols For Cleaning Up
# And Optimizing This Code, Making It More Flexible!
# If You've Found This Code Useful, Please Let Me Know.
# Visit My Site At nehe.gamedev.net
#
# Modified for Tcl3D by Paul Obermeier 2006/01/25
# See www.tcl3d.org for the Tcl3D extension.

package require tcl3d

# Font to be used in the Tk listbox.
set gDemo(listFont) {-family {Courier} -size 10}

# Obtain the name of this script file.
set gDemo(scriptFile) [info script]

# Display mode.
set gDemo(fullScreen) false

# Window size.
set gDemo(winWidth)  640
set gDemo(winHeight) 480

set gDemo(rtri)  0.0           ; # Angle For The Triangle ( NEW )
set gDemo(rquad) 0.0           ; # Angle For The Quad ( NEW )

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    ExitProg
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc SetFullScreenMode { win } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win $sw $sh
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+0" $sw $sh]
    wm geometry $win $fmtStr
    wm overrideredirect $win 1
    focus -force $win
}

proc SetWindowMode { win w h } {
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth  $win]

    wm minsize $win 10 10
    wm maxsize $win $sw $sh
    set fmtStr [format "%dx%d+0+25" $w $h]
    wm geometry $win $fmtStr
    wm overrideredirect $win 0
    focus -force $win
}

# Toggle between windowing and fullscreen mode.
proc ToggleWindowMode {} {
    if { $::gDemo(fullScreen) } {
        SetFullScreenMode .
        set ::gDemo(fullScreen) false
    } else {
        SetWindowMode . $::gDemo(winWidth) $::gDemo(winHeight)
        set ::gDemo(fullScreen) true
    }
}

# Resize And Initialize The GL Window
proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h        ; # Reset The Current Viewport
    glMatrixMode GL_PROJECTION  ; # Select The Projection Matrix
    glLoadIdentity              ; # Reset The Projection Matrix

    # Calculate The Aspect Ratio Of The Window
    gluPerspective 45.0 [expr double($w)/double($h)] 0.1 100.0

    glMatrixMode GL_MODELVIEW   ; # Select The Modelview Matrix
    glLoadIdentity              ; # Reset The Modelview Matrix
    set ::gDemo(winWidth)  $w
    set ::gDemo(winHeight) $h
}

# All Setup For OpenGL Goes Here
proc CreateCallback { toglwin } {
    glShadeModel GL_SMOOTH                  ; # Enable Smooth Shading
    glClearColor 0.0 0.0 0.0 0.5            ; # Black Background
    glClearDepth 1.0                        ; # Depth Buffer Setup
    glEnable GL_DEPTH_TEST                  ; # Enables Depth Testing
    glDepthFunc GL_LEQUAL                   ; # The Type Of Depth Testing To Do
    glHint GL_PERSPECTIVE_CORRECTION_HINT GL_NICEST ; # Really Nice Perspective Calculations
}

proc Animate {} {
    .fr.toglwin postredisplay
    set ::animateId [tcl3dAfterIdle Animate]
}

proc StartAnimation {} {
    if { ! [info exists ::animateId] } {
        Animate
    }
}

proc StopAnimation {} {
    if { [info exists ::animateId] } {
        after cancel $::animateId 
        unset ::animateId
    }
}

# Here's Where We Do All The Drawing
proc DisplayCallback { toglwin } {
    # Clear Screen And Depth Buffer
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT] 

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glLoadIdentity                      ; # Reset The Current Modelview Matrix
    glTranslatef -1.5 0.0 -6.0          ; # Move Left 1.5 Units And Into The Screen 6.0
    glRotatef $::gDemo(rtri) 0.0 1.0 0.0       ; # Rotate The Triangle On The Y axis ( NEW )
    glBegin GL_TRIANGLES                ; # Drawing Using Triangles
        glColor3f   1.0 0.0 0.0         ; # Red
        glVertex3f  0.0 1.0 0.0         ; # Top Of Triangle (Front)
        glColor3f   0.0 1.0 0.0         ; # Green
        glVertex3f -1.0 -1.0 1.0        ; # Left Of Triangle (Front)
        glColor3f   0.0 0.0 1.0         ; # Blue
        glVertex3f  1.0 -1.0 1.0        ; # Right Of Triangle (Front)
        glColor3f   1.0 0.0 0.0         ; # Red
        glVertex3f  0.0 1.0 0.0         ; # Top Of Triangle (Right)
        glColor3f   0.0 0.0 1.0         ; # Blue
        glVertex3f  1.0 -1.0 1.0        ; # Left Of Triangle (Right)
        glColor3f   0.0 1.0 0.0         ; # Green
        glVertex3f  1.0 -1.0 -1.0       ; # Right Of Triangle (Right)
        glColor3f   1.0 0.0 0.0         ; # Red
        glVertex3f  0.0 1.0 0.0         ; # Top Of Triangle (Back)
        glColor3f   0.0 1.0 0.0         ; # Green
        glVertex3f  1.0 -1.0 -1.0       ; # Left Of Triangle (Back)
        glColor3f   0.0 0.0 1.0         ; # Blue
        glVertex3f -1.0 -1.0 -1.0       ; # Right Of Triangle (Back)
        glColor3f   1.0 0.0 0.0         ; # Red
        glVertex3f  0.0 1.0 0.0         ; # Top Of Triangle (Left)
        glColor3f   0.0 0.0 1.0         ; # Blue
        glVertex3f -1.0 -1.0 -1.0       ; # Left Of Triangle (Left)
        glColor3f   0.0 1.0 0.0         ; # Green
        glVertex3f -1.0 -1.0 1.0        ; # Right Of Triangle (Left)
    glEnd                               ; # Done Drawing The Pyramid
    glLoadIdentity                      ; # Reset The Current Modelview Matrix
    glTranslatef 1.5 0.0 -7.0           ; # Move Right 1.5 Units And Into The Screen 7.0
    glRotatef $::gDemo(rquad) 1.0 1.0 1.0      ; # Rotate The Quad On The X axis ( NEW )

    glBegin GL_QUADS                    ; # Draw A Quad
        glColor3f 0.0 1.0 0.0           ; # Set The Color To Green
        glVertex3f  1.0  1.0 -1.0       ; # Top Right Of The Quad (Top)
        glVertex3f -1.0  1.0 -1.0       ; # Top Left Of The Quad (Top)
        glVertex3f -1.0  1.0  1.0       ; # Bottom Left Of The Quad (Top)
        glVertex3f  1.0  1.0  1.0       ; # Bottom Right Of The Quad (Top)
        glColor3f 1.0 0.5 0.0           ; # Set The Color To Orange
        glVertex3f  1.0 -1.0  1.0       ; # Top Right Of The Quad (Bottom)
        glVertex3f -1.0 -1.0  1.0       ; # Top Left Of The Quad (Bottom)
        glVertex3f -1.0 -1.0 -1.0       ; # Bottom Left Of The Quad (Bottom)
        glVertex3f  1.0 -1.0 -1.0       ; # Bottom Right Of The Quad (Bottom)
        glColor3f 1.0 0.0 0.0           ; # Set The Color To Red
        glVertex3f  1.0  1.0  1.0       ; # Top Right Of The Quad (Front)
        glVertex3f -1.0  1.0  1.0       ; # Top Left Of The Quad (Front)
        glVertex3f -1.0 -1.0  1.0       ; # Bottom Left Of The Quad (Front)
        glVertex3f  1.0 -1.0  1.0       ; # Bottom Right Of The Quad (Front)
        glColor3f 1.0 1.0 0.0           ; # Set The Color To Yellow
        glVertex3f  1.0 -1.0 -1.0       ; # Top Right Of The Quad (Back)
        glVertex3f -1.0 -1.0 -1.0       ; # Top Left Of The Quad (Back)
        glVertex3f -1.0  1.0 -1.0       ; # Bottom Left Of The Quad (Back)
        glVertex3f  1.0  1.0 -1.0       ; # Bottom Right Of The Quad (Back)
        glColor3f 0.0 0.0 1.0           ; # Set The Color To Blue
        glVertex3f -1.0  1.0  1.0       ; # Top Right Of The Quad (Left)
        glVertex3f -1.0  1.0 -1.0       ; # Top Left Of The Quad (Left)
        glVertex3f -1.0 -1.0 -1.0       ; # Bottom Left Of The Quad (Left)
        glVertex3f -1.0 -1.0  1.0       ; # Bottom Right Of The Quad (Left)
        glColor3f 1.0 0.0 1.0           ; # Set The Color To Violet
        glVertex3f  1.0  1.0 -1.0       ; # Top Right Of The Quad (Right)
        glVertex3f  1.0  1.0  1.0       ; # Top Left Of The Quad (Right)
        glVertex3f  1.0 -1.0  1.0       ; # Bottom Left Of The Quad (Right)
        glVertex3f  1.0 -1.0 -1.0       ; # Bottom Right Of The Quad (Right)
    glEnd
    set ::gDemo(rtri)  [expr $::gDemo(rtri) + 0.2]    ; # Increase The Rotation Variable For The Triangle ( NEW )
    set ::gDemo(rquad) [expr $::gDemo(rquad) -0.15]   ; # Decrease The Rotation Variable For The Quad ( NEW )
    
    $toglwin swapbuffers
}

proc Cleanup {} {
    uplevel #0 unset gDemo
}

# Put all exit related code here.
proc ExitProg {} {
    exit
}

# Create a PDF file of the window contents.
proc CreatePdf { toglwin } {
    if { [tcl3dHaveGl2ps] } {
        set fileName [format "%s.%s" [file rootname $::gDemo(scriptFile)] "pdf"]
        # Create a name on the file system, if running from within a Starpack.
        set fileName [tcl3dGenExtName $fileName]
        tcl3dGl2psCreatePdf $toglwin $fileName "[wm title .]"
    } else {
        tk_messageBox -icon info -type ok -title "Info" \
                      -message "PDF creation needs the gl2ps extension.\n\
                                Available in Tcl3D versions greater than 0.3."
    }
    $toglwin postredisplay
}

# Create the OpenGL window and some Tk helper widgets.
proc CreateWindow {} {
    frame .fr
    pack .fr -expand 1 -fill both
    # Create Our OpenGL Window
    togl .fr.toglwin -width $::gDemo(winWidth) -height $::gDemo(winHeight) \
                     -swapinterval 1 \
                     -double true -depth true \
                     -createcommand CreateCallback \
                     -reshapecommand ReshapeCallback \
                     -displaycommand DisplayCallback 
    listbox .fr.usage -font $::gDemo(listFont) -height 4
    label   .fr.info
    grid .fr.toglwin -row 0 -column 0 -sticky news
    grid .fr.usage   -row 1 -column 0 -sticky news
    grid .fr.info    -row 2 -column 0 -sticky news
    grid rowconfigure .fr 0 -weight 1
    grid columnconfigure .fr 0 -weight 1
    wm title . "Tcl3D demo: NeHe's Solid Object Tutorial (Lesson 5)"

    # Watch For ESC Key And Quit Messages
    wm protocol . WM_DELETE_WINDOW "ExitProg"
    bind . <Key-Escape> "ExitProg"
    bind . <Key-F1>     "ToggleWindowMode"
    bind . <Key-F12>    "CreatePdf .fr.toglwin"

    bind .fr.toglwin <1> "StartAnimation"
    bind .fr.toglwin <2> "StopAnimation"
    bind .fr.toglwin <3> "StopAnimation"
    bind .fr.toglwin <Control-Button-1> "StopAnimation"

    .fr.usage insert end "Key-Escape Exit"
    .fr.usage insert end "Key-F1     Toggle window mode"
    .fr.usage insert end "Key-F12    Create PDF file"
    .fr.usage insert end "Mouse-L|MR Start|Stop animation"

    .fr.usage configure -state disabled
}

CreateWindow
PrintInfo [tcl3dOglGetInfoString]
if { [file tail [info script]] eq [file tail $::argv0] } {
    # If started directly from tclsh or wish, then start animation.
    update
    StartAnimation
}
