#-----------------------------------------------------------------------------
#           Name: ogl_near_far_clip.cpp
#         Author: Kevin Harris (kevin@codesampler.com)
#  Last Modified: 02/01/05
#    Description: This sample demonstrates how adjustments to OpenGL's 
#                 near and far clip planes effect the view.
#
#   Control Keys: Up         - View moves forward
#                 Down       - View moves backward
#                 Left       - View strafes left
#                 Right      - View strafes Right
#                 Left Mouse - Perform looking
#                 Mouse      - Look about the scene
#
#                 F1         - Increase near clip value
#                 F2         - Decrease near clip value
#                 F3         - Increase far clip value
#                 F4         - Decrease far clip value
#-----------------------------------------------------------------------------
#
# Original C++ code by Kevin Harris (kevin@codesampler.com)
# See www.codesampler.com for the original files
# OpenGL samples page 2: Near/Far Clipping Plane
#
# Modified for Tcl3D by Paul Obermeier 2007/03/10
# See www.tcl3d.org for the Tcl3D extension.

package require Tk
package require tcl3d

# Font to be used in the Tk listbox.
set g_listFont {-family {Courier} -size 10}

set g_WinWidth  640
set g_WinHeight 480

set g_fMoveSpeed 25.0

set g_vEye   [tcl3dVectorFromArgs GLfloat 7.0 4.0 7.0]     ; # Eye Position
set g_vLook  [tcl3dVectorFromArgs GLfloat -0.5 -0.4 -0.5]  ; # Look Vector
set g_vUp    [tcl3dVectorFromArgs GLfloat 0.0 1.0 0.0]     ; # Up Vector
set g_vRight [tcl3dVectorFromArgs GLfloat 1.0 0.0 0.0]     ; # Right Vector

set g_StopWatch [tcl3dNewSwatch]
tcl3dStartSwatch $g_StopWatch
set ::g_LastMousePosX(1) 0
set ::g_LastMousePosY(1) 0

set g_fNearClipPlane  8.0
set g_fFarClipPlane  20.0
set g_bNearFarClipPlanesAdjusted false

set g_tmpMat [tcl3dVector GLfloat 16]
set g_tmpVec [tcl3dVector GLfloat 3]

# Mesh properties...
set g_nNumVertsAlongX   32
set g_nNumVertsAlongZ   32
set g_fMeshLengthAlongX 10.0
set g_fMeshLengthAlongZ 10.0

# GL_N3F_V3F
set NX 0
set NY 1
set NZ 2
set X  3
set Y  4
set Z  5
set SIZE 6

set g_nMeshVertCount [expr {($g_nNumVertsAlongX-1) * \
                            ($g_nNumVertsAlongZ-1) * 6}]

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc SetMouseInput { x y } {
    set ::g_LastMousePosX(1) $x
    set ::g_LastMousePosY(1) $y
}

proc GetMouseInput { x y } {
    set nXDiff [expr {$x - $::g_LastMousePosX(1)}]
    set nYDiff [expr {$y - $::g_LastMousePosY(1)}]
        
    if { $nYDiff != 0 } {
        tcl3dMatfRotate [expr {-1.0*$nYDiff / 3.0}] $::g_vRight $::g_tmpMat
        tcl3dMatfTransformVector $::g_vLook $::g_tmpMat $::g_vLook
        tcl3dMatfTransformVector $::g_vUp $::g_tmpMat $::g_vUp
    }

    if { $nXDiff != 0 } {
        set vec [tcl3dVectorFromArgs GLfloat 0 1 0]
        tcl3dMatfRotate [expr {-1.0*$nXDiff / 3.0}] $vec $::g_tmpMat
        tcl3dMatfTransformVector $::g_vLook $::g_tmpMat $::g_vLook
        tcl3dMatfTransformVector $::g_vUp $::g_tmpMat $::g_vUp
        $vec delete
    }

    set ::g_LastMousePosX(1) $x
    set ::g_LastMousePosY(1) $y
    .fr.toglwin postredisplay
}

proc MoveForward {} {
    tcl3dVec3fScale [expr {-1.0 * $::g_fMoveSpeed * $::g_fElapsedTime}] \
                    $::g_vLook $::g_tmpVec
    tcl3dVec3fSubtract $::g_vEye $::g_tmpVec $::g_vEye
    .fr.toglwin postredisplay
}

proc MoveBackward {} {
    tcl3dVec3fScale [expr {-1.0 * $::g_fMoveSpeed * $::g_fElapsedTime}] \
                    $::g_vLook $::g_tmpVec
    tcl3dVec3fAdd $::g_vEye $::g_tmpVec $::g_vEye
    .fr.toglwin postredisplay
}

proc MoveLeft {} {
    tcl3dVec3fScale [expr {$::g_fMoveSpeed * $::g_fElapsedTime}] \
                    $::g_vRight $::g_tmpVec
    tcl3dVec3fSubtract $::g_vEye $::g_tmpVec $::g_vEye
    .fr.toglwin postredisplay
}

proc MoveRight {} {
    tcl3dVec3fScale [expr {$::g_fMoveSpeed * $::g_fElapsedTime}] \
                    $::g_vRight $::g_tmpVec
    tcl3dVec3fAdd $::g_vEye $::g_tmpVec $::g_vEye
    .fr.toglwin postredisplay
}

proc MoveUp {} {
    $::g_tmpVec set 0 0
    $::g_tmpVec set 1 [expr {$::g_fMoveSpeed * $::g_fElapsedTime}]
    $::g_tmpVec set 2 0
    tcl3dVec3fAdd $::g_vEye $::g_tmpVec $::g_vEye
    .fr.toglwin postredisplay
}

proc MoveDown {} {
    $::g_tmpVec set 0 0
    $::g_tmpVec set 1 [expr {$::g_fMoveSpeed * $::g_fElapsedTime}]
    $::g_tmpVec set 2 0
    tcl3dVec3fSubtract $::g_vEye $::g_tmpVec $::g_vEye
    .fr.toglwin postredisplay
}

proc AdjustNearClipPlane { toglwin val } {
    set ::g_fNearClipPlane [expr {$::g_fNearClipPlane + $val}]
    set ::g_bNearFarClipPlanesAdjusted true
    $toglwin postredisplay
}

proc AdjustFarClipPlane { toglwin val } {
    set ::g_fFarClipPlane [expr {$::g_fFarClipPlane + $val}]
    set ::g_bNearFarClipPlanesAdjusted true
    $toglwin postredisplay
}

proc CreateMesh {} {
    # Compute position deltas for moving down the X, and Z axis during mesh 
    # creation
    set dX [expr { (1.0 / ($::g_nNumVertsAlongX-1))}]
    set dZ [expr {-(1.0 / ($::g_nNumVertsAlongZ-1))}]

    # These are all the same...
    #              NX  NY  NZ   X   Y   Z
    set constList {0.0 1.0 0.0  0.0 0.0 0.0}
    for {set i 0 } { $i < $::g_nMeshVertCount } { incr i } {
        lappend meshList $constList
    }
    # Make a flat list.
    set meshList [join $meshList]

    # Create all the vertex points required by the mesh...
    # Note: Mesh tesselation occurs in the X,Z plane.

    # For each row of our mesh...
    set i 0
    for {set z 0 } { $z < [expr {$::g_nNumVertsAlongZ-1}] } { incr z } {
        # Fill the row with quads which are composed of two triangles each...
        for {set x 0 } { $x < [expr {$::g_nNumVertsAlongX-1}] } { incr x } {
            # First triangle of the current quad
            #   ___ 2
            #  |  /|
            #  |/__|
            # 0     1

            # 0
            set off [expr {$i*$::SIZE}]
            lset meshList [expr {$off + $::X}] \
                          [expr {$::g_fMeshLengthAlongX * $x * $dX}]
            lset meshList [expr {$off + $::Z}] \
                          [expr {$::g_fMeshLengthAlongZ * $z * $dZ}]
            incr i

            # 1
            set off [expr {$i*$::SIZE}]
            lset meshList [expr {$off + $::X}] \
                          [expr {$::g_fMeshLengthAlongX * ($x+1.0) * $dX}]
            lset meshList [expr {$off + $::Z}] \
                          [expr {$::g_fMeshLengthAlongZ * $z * $dZ}]
            incr i

            # 2
            set off [expr {$i*$::SIZE}]
            lset meshList [expr {$off + $::X}] \
                          [expr {$::g_fMeshLengthAlongX * ($x+1.0) * $dX}]
            lset meshList [expr {$off + $::Z}] \
                          [expr {$::g_fMeshLengthAlongZ * ($z+1.0) * $dZ}]
            incr i

            # Second triangle of the current quad
            # 2 ___ 1
            #  |  /|
            #  |/__|
            # 0

            # 0
            set off [expr {$i*$::SIZE}]
            lset meshList [expr {$off + $::X}] \
                          [expr {$::g_fMeshLengthAlongX * $x * $dX}]
            lset meshList [expr {$off + $::Z}] \
                          [expr {$::g_fMeshLengthAlongZ * $z * $dZ}]
            incr i

            # 1
            set off [expr {$i*$::SIZE}]
            lset meshList [expr {$off + $::X}] \
                          [expr {$::g_fMeshLengthAlongX * ($x+1.0) * $dX}]
            lset meshList [expr {$off + $::Z}] \
                          [expr {$::g_fMeshLengthAlongZ * ($z+1.0) * $dZ}]
            incr i

            # 2
            set off [expr {$i*$::SIZE}]
            lset meshList [expr {$off + $::X}] \
                          [expr {$::g_fMeshLengthAlongX * $x * $dX}]
            lset meshList [expr {$off + $::Z}] \
                          [expr {$::g_fMeshLengthAlongZ * ($z+1.0) * $dZ}]
            incr i
        }
    }
    set ::g_meshVertices [tcl3dVectorFromList GLfloat $meshList]
}

proc CreateCallback { toglwin } {
    glClearColor 0.35 0.53 0.7 1.0

    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluPerspective 45.0 [expr {double($::g_WinWidth) / double($::g_WinHeight)}] \
                   $::g_fNearClipPlane $::g_fFarClipPlane

    CreateMesh
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluPerspective 45.0 [expr {double($w)/double($h)}] \
                   $::g_fNearClipPlane $::g_fFarClipPlane
}

#-----------------------------------------------------------------------------
# Name : UpdateViewMatrix
# Desc : Builds a view matrix suitable for OpenGL.
#
# Here's what the final view matrix should look like:
#
#  |  rx   ry   rz  -(r.e) |
#  |  ux   uy   uz  -(u.e) |
#  | -lx  -ly  -lz   (l.e) |
#  |   0    0    0     1   |
#
# Where r = Right vector
#       u = Up vector
#       l = Look vector
#       e = Eye position in world space
#       . = Dot-product operation
#
#-----------------------------------------------------------------------------

proc UpdateViewMatrix {} {
    tcl3dMatfIdentity $::g_tmpMat

    tcl3dVec3fNormalize $::g_vLook

    tcl3dVec3fCrossProduct $::g_vLook $::g_vUp $::g_vRight
    tcl3dVec3fNormalize $::g_vRight

    tcl3dVec3fCrossProduct $::g_vRight $::g_vLook $::g_vUp
    tcl3dVec3fNormalize $::g_vUp

    set rightX [$::g_vRight get 0]
    set rightY [$::g_vRight get 1]
    set rightZ [$::g_vRight get 2]
    set upX    [$::g_vUp get 0]
    set upY    [$::g_vUp get 1]
    set upZ    [$::g_vUp get 2]
    set lookX  [$::g_vLook get 0]
    set lookY  [$::g_vLook get 1]
    set lookZ  [$::g_vLook get 2]

    $::g_tmpMat set  0 $rightX
    $::g_tmpMat set  1 $upX
    $::g_tmpMat set  2 [expr {-1.0 * $lookX}]
    $::g_tmpMat set  3 0

    $::g_tmpMat set  4 $rightY
    $::g_tmpMat set  5 $upY
    $::g_tmpMat set  6 [expr {-1.0 * $lookY}]
    $::g_tmpMat set  7 0

    $::g_tmpMat set  8 $rightZ
    $::g_tmpMat set  9 $upZ
    $::g_tmpMat set 10 [expr {-1.0 * $lookZ}]
    $::g_tmpMat set 11 0

    $::g_tmpMat set 12 [expr {-1.0 * [tcl3dVec3fDotProduct $::g_vRight $::g_vEye]}]
    $::g_tmpMat set 13 [expr {-1.0 * [tcl3dVec3fDotProduct $::g_vUp    $::g_vEye]}]
    $::g_tmpMat set 14 [expr { 1.0 * [tcl3dVec3fDotProduct $::g_vLook  $::g_vEye]}]
    $::g_tmpMat set 15 1

    set viewList [tcl3dVectorToList $::g_tmpMat 16]
    glMultMatrixf $viewList
}

proc DisplayCallback { toglwin } {
    set startTime [tcl3dLookupSwatch $::g_StopWatch]

    glClear [expr {$::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT}]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    # Set near/far clip planes...
    if { $::g_bNearFarClipPlanesAdjusted == true } {
        glMatrixMode GL_PROJECTION
        glLoadIdentity
        gluPerspective 45.0 [expr {double($::g_WinWidth) / double($::g_WinHeight)}] \
                       $::g_fNearClipPlane $::g_fFarClipPlane
        set ::g_bNearFarClipPlanesAdjusted false
    }

    glMatrixMode GL_MODELVIEW
    glLoadIdentity
    UpdateViewMatrix

    # Render floor...
    glColor3f 1.0 1.0 1.0
    glPolygonMode GL_FRONT GL_LINE

    glPushMatrix
        glScalef 2.0 2.0 2.0
        glTranslatef -5.0 -0.8 5.0
        
        glInterleavedArrays GL_N3F_V3F 0 $::g_meshVertices
        glDrawArrays GL_TRIANGLES 0 $::g_nMeshVertCount
    glPopMatrix

    # Render teapot...
    glColor3f 0.0 0.0 1.0
    glutWireTeapot 2.0

    $toglwin swapbuffers
    set endTime [tcl3dLookupSwatch $::g_StopWatch]
    set ::g_fElapsedTime [expr {$endTime - $startTime}]
}

proc Cleanup {} {
    $::g_tmpMat delete
    $::g_tmpVec delete
    $::g_meshVertices delete
    tcl3dDeleteSwatch $::g_StopWatch

    foreach var [info globals g_*] {
        uplevel #0 unset $var
    }
}

proc ExitProg {} {
    exit
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width $g_WinWidth -height $g_WinHeight \
                 -double true -depth true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::g_listFont -height 6
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: CodeSampler's Near/Far Clip Plane"

# Watch For ESC Key And Quit Messages
wm protocol . WM_DELETE_WINDOW "ExitProg"
bind . <Key-Escape> "ExitProg"
bind . <Key-F1>     "AdjustNearClipPlane .fr.toglwin  0.1"
bind . <Key-F2>     "AdjustNearClipPlane .fr.toglwin -0.1"
bind . <Key-F3>     "AdjustFarClipPlane  .fr.toglwin  0.1"
bind . <Key-F4>     "AdjustFarClipPlane  .fr.toglwin -0.1"
bind . <Key-Up>     "MoveForward"
bind . <Key-Down>   "MoveBackward"
bind . <Key-Left>   "MoveLeft"
bind . <Key-Right>  "MoveRight"
bind . <Key-Home>   "MoveUp"
bind . <Key-End>    "MoveDown"
bind .fr.toglwin <1>         "SetMouseInput %x %y"
bind .fr.toglwin <B1-Motion> "GetMouseInput %x %y"

.fr.usage insert end "Key-Escape     Exit"
.fr.usage insert end "Key-F1|F2      Increase|Decrease near clip plane"
.fr.usage insert end "Key-F3|F4      Increase|Decrease far clip plane"
.fr.usage insert end "Key-Up|Down    View moves forward|backward"
.fr.usage insert end "Key-Left|Right View strafes to the left|right"
.fr.usage insert end "Key-Home|End   View elevates up|down"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
