# checkerBoard.tcl
#
# This program creates a checkerboard image in two ways.
# The first texture is created with an algorithm, as used in some of the
# RedBook examples (ex. checker.tcl). This algorithm has been converted 1:1
# from C to Tcl. Very slow.
# The second image is created using the Img extension, which is essentially 
# faster.
#
# Author: Paul Obermeier
# Date: 2006-09-22

package require tcl3d
package require Img

# Create checkerboard texture
set g_TexWidth  64
set g_TexHeight 64

# Font to be used in the Tk listbox.
set g_ListFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# The classical method converted 1:1 from C to Tcl. Very slow.
proc makeCheckImage1 {} {
    set ::g_CheckImg1 [tcl3dVector GLubyte [expr $::g_TexHeight*$::g_TexWidth*4]]

    for { set i 0 } { $i < $::g_TexHeight } { incr i } {
        for { set j 0 } { $j < $::g_TexWidth } { incr j } { 
         set c [expr {(((($i&0x8)==0)^(($j&0x8))==0))*255}]
         $::g_CheckImg1 set [expr {($i*$::g_TexWidth + $j)*4 + 0}] $c
         $::g_CheckImg1 set [expr {($i*$::g_TexWidth + $j)*4 + 1}] $c
         $::g_CheckImg1 set [expr {($i*$::g_TexWidth + $j)*4 + 2}] $c
         $::g_CheckImg1 set [expr {($i*$::g_TexWidth + $j)*4 + 3}] 255
      }
   }
}

# Create the checker image with the use of the Img extension. Fast.
proc makeCheckImage2 {} {
    set tmp [image create photo -width 2 -height 2]
    set img [image create photo -width $::g_TexWidth -height $::g_TexHeight]
    $tmp put {{#ffffff #FF0000} {#FF0000 #ffffff}} 
    $img copy $tmp -from 0 0 2 2 -zoom 8 8 -to 0 0 $::g_TexWidth $::g_TexHeight
    set ::g_CheckImg2 [tcl3dVectorFromPhoto $img]
    image delete $tmp
    image delete $img
}

proc CreateCallback { toglwin } {    
    glClearColor 0.5 0.5 0.5 0.0
    glShadeModel GL_FLAT
    glEnable GL_DEPTH_TEST
}

proc CreateCheckerImgs {} {
    set t1 [time makeCheckImage1]
    set t2 [time makeCheckImage2]
    .fr.usage insert end "Black checker: $t1"
    .fr.usage insert end "Red   checker: $t2"

    glPixelStorei GL_UNPACK_ALIGNMENT 1

    set ::g_TexName [tcl3dVector GLuint 2]
    glGenTextures 2 $::g_TexName

    glBindTexture GL_TEXTURE_2D [$::g_TexName get 0]
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_WRAP_S $::GL_REPEAT
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_WRAP_T $::GL_REPEAT
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MAG_FILTER $::GL_NEAREST
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MIN_FILTER $::GL_NEAREST

    glTexImage2D GL_TEXTURE_2D 0 $::GL_RGBA \
                 $::g_TexWidth $::g_TexHeight \
                 0 GL_RGBA GL_UNSIGNED_BYTE $::g_CheckImg1

    glBindTexture GL_TEXTURE_2D [$::g_TexName get 1]
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_WRAP_S $::GL_REPEAT
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_WRAP_T $::GL_REPEAT
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MAG_FILTER $::GL_NEAREST
    glTexParameteri GL_TEXTURE_2D GL_TEXTURE_MIN_FILTER $::GL_NEAREST

    glTexImage2D GL_TEXTURE_2D 0 $::GL_RGBA \
                 $::g_TexWidth $::g_TexHeight \
                 0 GL_RGBA GL_UNSIGNED_BYTE $::g_CheckImg2
}

proc DisplayCallback { toglwin } {
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glEnable GL_TEXTURE_2D
    glTexEnvf GL_TEXTURE_ENV GL_TEXTURE_ENV_MODE $::GL_DECAL

    glBindTexture GL_TEXTURE_2D [$::g_TexName get 0]
    glBegin GL_QUADS
        glTexCoord2f 0.0 0.0 ; glVertex3f -2.0 -1.0 0.0
        glTexCoord2f 0.0 1.0 ; glVertex3f -2.0  1.0 0.0
        glTexCoord2f 1.0 1.0 ; glVertex3f  0.0  1.0 0.0
        glTexCoord2f 1.0 0.0 ; glVertex3f  0.0 -1.0 0.0
    glEnd

    glBindTexture GL_TEXTURE_2D [$::g_TexName get 1]
    glBegin GL_QUADS
        glTexCoord2f 0.0 0.0 ; glVertex3f 0.0 -1.0 0.0
        glTexCoord2f 0.0 1.0 ; glVertex3f 0.0 1.0 0.0
        glTexCoord2f 1.0 1.0 ; glVertex3f 2.0 1.0 0.0
        glTexCoord2f 1.0 0.0 ; glVertex3f 2.0 -1.0 0.0
    glEnd
    glFlush
    glDisable GL_TEXTURE_2D
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    gluPerspective 60.0 [expr double($w)/double($h)] 1.0 30.0
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
    glTranslatef 0.0 0.0 -3.6
}

proc Cleanup {} {
    if { [info exists ::g_TexName] } {
        glDeleteTextures 1 [$::g_TexName get 0]
        glDeleteTextures 1 [$::g_TexName get 1]
        $::g_TexName delete
    }
    if { [info exists ::g_CheckImg1] } {
        $::g_CheckImg1 delete
    }
    if { [info exists ::g_CheckImg2] } {
        $::g_CheckImg2 delete
    }
    foreach var [info globals g_*] {
        uplevel #0 unset $var
    }
}

proc ExitProg {} {
    exit
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 250 -height 250 -double true -depth true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -font $::g_ListFont -height 3
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: Texture generation comparison"

wm protocol . WM_DELETE_WINDOW "ExitProg"
bind . <Key-Escape> "ExitProg"

.fr.usage insert end "Key-Escape Exit"

CreateCheckerImgs
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
