# AutoTransform.tcl
#
# Original C++ code by Peter Wraae Marino and Michael Bach Jensen.
# See www.osghelp.com for the original files.
#
# Modified for Tcl3D by Paul Obermeier 2010/03/20.
# See www.tcl3d.org for the Tcl3D extension.

package require tcl3d

if { ! [tcl3dHaveOsg] } {
    tk_messageBox -icon error -type ok -title "Missing Tcl3D module" \
                  -message "Demo needs the tcl3dOSG module."
    proc Cleanup {} {}
    exit 1
    return
}

# Font to be used in the Tk listbox.
set gDemo(listFont) {-family {Courier} -size 10}

# Window size.
set gDemo(winWidth)  640
set gDemo(winHeight) 480

# Determine the directory and filename of this script.
set gDemo(scriptFile) [info script]
set gDemo(scriptDir) [file dirname $gDemo(scriptFile)]

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    puts "Error: $msg\n\n$::errorInfo"
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    ExitProg
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# Idle callback to redisplay the scene.
proc Animate {} {
    .fr.toglwin postredisplay
    set ::animateId [tcl3dAfterIdle Animate]
}

proc StartAnimation {} {
    if { ! [info exists ::animateId] } {
        Animate
    }
}

proc StopAnimation {} {
    if { [info exists ::animateId] } {
        after cancel $::animateId 
        unset ::animateId
    }
}

proc CreateCallback { toglwin } {
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    global gDemo

    set w [$toglwin width]
    set h [$toglwin height]

    # Propagate resize event to embedded OSG window.
    tcl3dOsgWindowResize $toglwin [tcl3dOsgGetOsgWin] $w $h
    set gDemo(winWidth)  $w
    set gDemo(winHeight) $h
}

proc DisplayCallback { toglwin } {
    if { [viewer valid] } {
        viewer frame
    }
    $toglwin swapbuffers
}

proc Cleanup {} {
    uplevel #0 unset gDemo
    viewer -delete
}

proc ExitProg {} {
    exit
}

proc SaveOsgToFile {} {
    global gDemo

    set osgRoot [viewer getSceneData]
    set outFile [format "%s.osgt" [file rootname $gDemo(scriptFile)]]
    # Create a name on the file system, if running from within a Starpack.
    set outFile [tcl3dGenExtName $outFile]
    puts "Saving scenegraph to file $outFile"

    if { ! [osgDB::writeNodeFile $osgRoot $outFile] } {
        puts "Failed to write scenegraph to file $outFile"
    }
}

proc CreateWidgets { osgwin } {
    global gDemo

    frame .fr
    pack .fr -expand 1 -fill both
    set toglwin .fr.toglwin
    togl $toglwin -width $gDemo(winWidth) -height $gDemo(winHeight) \
                  -double true -depth true -alpha true \
                  -createcommand CreateCallback \
                  -reshapecommand ReshapeCallback \
                  -displaycommand DisplayCallback 
    listbox .fr.usage -font $gDemo(listFont) -height 3
    label   .fr.info
    grid $toglwin   -row 0 -column 0 -sticky news
    grid .fr.usage  -row 1 -column 0 -sticky news
    grid .fr.info   -row 2 -column 0 -sticky news
    grid rowconfigure .fr 0 -weight 1
    grid columnconfigure .fr 0 -weight 1
    wm title . "Tcl3D demo: OsgHelp tutorial AutoTransform"

    wm protocol . WM_DELETE_WINDOW "ExitProg"
    bind . <Key-Escape> "ExitProg"
    bind . <Key-f>      "SaveOsgToFile"

    # Propagate key and mouse events to embedded OSG window.
    bind . <KeyPress> "tcl3dOsgKeyPress $toglwin $osgwin %N"

    tcl3dOsgAddTrackballBindings $toglwin $osgwin

    .fr.usage insert end "Key-Escape Exit"
    .fr.usage insert end "Key-f      Save SceneGraph to file"
    .fr.usage insert end "Mouse      Trackball"

    .fr.usage configure -state disabled
}

#
# Start of tutorial specific code.
#

proc CreateBox { width depth height } {
    osg::Geode geode
    osg::Box box [osg::Vec3 v1 0.0 0.0 -$height] $width $depth $height
    geode addDrawable [osg::ShapeDrawable myDrawable box]
    return geode
}

proc CreateMarker {} {
    # create drawable geometry object
    osg::Geometry geo

    # add 4 vertices, creating a square around 0, 0, 0
    osg::Vec3Array verts
    verts push [osg::Vec3 v1 -1  0  0]
    verts push [osg::Vec3 v1  0  1  0]
    verts push [osg::Vec3 v1  1  0  0]
    verts push [osg::Vec3 v1  0 -1  1]
    geo setVertexArray verts

    # create a primitive set (add index numbers) for drawing lines
    osg::DrawElementsUInt primitiveSet $::osg::PrimitiveSet_LINE_LOOP 0
    primitiveSet push 3
    primitiveSet push 2
    primitiveSet push 1
    primitiveSet push 0
    geo addPrimitiveSet primitiveSet

    return geo
}

proc CreateMarkerNode { { scale 1.0 } } {
    # create marker geode
    osg::Geode geode
    geode addDrawable [CreateMarker]

    # turn off lighting
    set pStateSet [geode getOrCreateStateSet]
    $pStateSet setMode $::GL_LIGHTING $::osg::StateAttribute_OFF

    # scale the marker
    osg::PositionAttitudeTransform pat
    pat setScale [osg::Vec3d v1 $scale $scale $scale]
    pat addChild geode

    return pat
}

proc CreateAutoTransform { x y z scale } {
    # setup an auto transform node so that it scales its child to screen size
    # and rotates it towards the camera
    osg::AutoTransform autoTransform

    autoTransform setAutoScaleToScreen true
    autoTransform setAutoRotateMode $::osg::AutoTransform_ROTATE_TO_CAMERA
    autoTransform addChild [CreateMarkerNode $scale]
    autoTransform setPosition [osg::Vec3d v1 $x $y $z]
    
    return autoTransform
}

proc CreateScene {} {
    osg::Group group

    # show a box that is 10 units to each side
    group addChild [CreateBox 10 10 0.1]

    # put auto transform scaled markers around each corner of the box
    set scale 10.0
    group addChild [CreateAutoTransform -5 -5 0 $scale]
    group addChild [CreateAutoTransform -5  5 0 $scale]
    group addChild [CreateAutoTransform  5 -5 0 $scale]
    group addChild [CreateAutoTransform  5  5 0 $scale]

    return group
}

# We have the geometry ready. Now create the viewer and the Tk widgets.
osgViewer::ViewerRef viewer [osgViewer::Viewer]

# set the scene-graph data the viewer will render
viewer setSceneData [CreateScene]

viewer setCameraManipulator [osgGA::TrackballManipulator]

if { $argc >= 1 && [lindex $argv 0] eq "-viewer" } {
    # Only use the standard OSG viewer window without any Tk widgets.
    viewer setUpViewInWindow 50 50 500 400
    viewer run
    exit 0
}

# Use the OSG viewer inside a Togl widget.
set osgwin [viewer setUpViewerAsEmbeddedInWindow 50 50 500 400]
tcl3dOsgSetOsgWin $osgwin

viewer realize

CreateWidgets $osgwin

PrintInfo [tcl3dOsgGetInfoString]

if { [file tail [info script]] eq [file tail $::argv0] } {
    # If started directly from tclsh or wish, then start animation.
    update
    StartAnimation
}
