# NPS10_TestManualCamera.tcl
#
# Original C++ code by Joseph Sullivan.
# See http://www.openscenegraph.org/projects/osg/wiki/Support/Tutorials
# for the original files.
#
# Modified for Tcl3D by Paul Obermeier 2009/05/01.
# See www.tcl3d.org for the Tcl3D extension.

package require tcl3d

if { ! [tcl3dHaveOsg] } {
    tk_messageBox -icon error -type ok -title "Missing Tcl3D module" \
                  -message "Demo needs the tcl3dOSG module."
    proc Cleanup {} {}
    exit 1
    return
}

# Font to be used in the Tk listbox.
set gDemo(listFont) {-family {Courier} -size 10}

# Window size.
set gDemo(winWidth)  640
set gDemo(winHeight) 480

# Determine the directory and filename of this script.
set gDemo(scriptFile) [info script]
set gDemo(scriptDir)  [file dirname $gDemo(scriptFile)]

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    puts "Error: $msg\n\n$::errorInfo"
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    ExitProg
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# Idle callback to redisplay the scene.
proc Animate {} {
    .fr.toglwin postredisplay
    set ::animateId [tcl3dAfterIdle Animate]
}

proc StartAnimation {} {
    if { ! [info exists ::animateId] } {
        Animate
    }
}

proc StopAnimation {} {
    if { [info exists ::animateId] } {
        after cancel $::animateId 
        unset ::animateId
    }
}

proc CreateCallback { toglwin } {
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    # Propagate resize event to embedded OSG window.
    tcl3dOsgWindowResize $toglwin [tcl3dOsgGetOsgWin] $w $h
}

proc DisplayCallback { toglwin } {
    global gManuallyPlaceCamera

    if { [viewer valid] } {
        if { $gManuallyPlaceCamera } {
            set matInv [osg::Matrixd_inverse myCameraMatrix]
            $matInv mult $matInv \
                         [osg::Matrixd_rotate [tcl3dDegToRad -90] 1 0 0]
            trackManip setByInverseMatrix $matInv
        }
        viewer frame
    }
    $toglwin swapbuffers
}

proc Cleanup {} {
    uplevel #0 unset gDemo
    viewer -delete
}

proc ExitProg {} {
    exit
}

proc SaveOsgToFile {} {
    global gDemo

    set osgRoot [viewer getSceneData]
    set outFile [format "%s.osgt" [file rootname $gDemo(scriptFile)]]
    # Create a name on the file system, if running from within a Starpack.
    set outFile [tcl3dGenExtName $outFile]
    puts "Saving scenegraph to file $outFile"

    if { ! [osgDB::writeNodeFile $osgRoot $outFile] } {
        puts "Failed to write scenegraph to file $outFile"
    }
}

proc CreateWidgets { osgwin } {
    global gDemo

    frame .fr
    pack .fr -expand 1 -fill both
    set toglwin .fr.toglwin
    togl $toglwin -width $gDemo(winWidth) -height $gDemo(winHeight) \
                  -double true -depth true -alpha true \
                  -createcommand CreateCallback \
                  -reshapecommand ReshapeCallback \
                  -displaycommand DisplayCallback 
    listbox .fr.usage -font $::gDemo(listFont) -height 4
    label   .fr.info
    grid $toglwin   -row 0 -column 0 -sticky news
    grid .fr.usage  -row 1 -column 0 -sticky news
    grid .fr.info   -row 2 -column 0 -sticky news
    grid rowconfigure .fr 0 -weight 1
    grid columnconfigure .fr 0 -weight 1
    wm title . "Tcl3D demo: Sullivan's OSG tutorial #10 (TestManualCamera)"

    wm protocol . WM_DELETE_WINDOW "ExitProg"
    bind . <Key-Escape> "ExitProg"
    bind . <Key-f>      "SaveOsgToFile"
    bind . <Key-v>      "ToggleView"

    # Propagate key and mouse events to embedded OSG window.
    bind . <KeyPress> "tcl3dOsgKeyPress $toglwin $osgwin %N"

    tcl3dOsgAddTrackballBindings $toglwin $osgwin

    .fr.usage insert end "Key-Escape Exit"
    .fr.usage insert end "Key-f      Save SceneGraph to file"
    .fr.usage insert end "Key-v      Toggle view mode"
    .fr.usage insert end "Mouse      Trackball"

    .fr.usage configure -state disabled
}

#
# Start of tutorial specific code.
#

set gManuallyPlaceCamera false

proc ToggleView {} {
    global gManuallyPlaceCamera

    if { ! $gManuallyPlaceCamera } {
        set gManuallyPlaceCamera true
    } else {
        set gManuallyPlaceCamera false
    }
}

osg::Group root
osg::PositionAttitudeTransform tankXform

osgViewer::ViewerRef viewer [osgViewer::Viewer]

set tankFile   [file join $::gDemo(scriptDir) "Data/Models/t72-tank/t72-tank_des.flt"]
set groundFile [file join $::gDemo(scriptDir) "Data/Models/JoeDirt/JoeDirt.flt"]
set tankNode   [osgDB::readNodeFile $tankFile]
set groundNode [osgDB::readNodeFile $groundFile]

# Create green Irish sky
osg::ClearNode backdrop
backdrop setClearColor [osg::Vec4 bgcol 0.0 0.8 0.0 1.0]
root addChild backdrop
root addChild $groundNode

osg::PositionAttitudeTransform tankXform
root addChild tankXform
tankXform addChild $tankNode

tankXform setPosition [osg::Vec3d pos 10 10 8]
tankXform setAttitude [osg::Quat att [tcl3dDegToRad -45.0] [osg::Vec3 axis 0 0 1]]

osgGA::TrackballManipulator trackManip
viewer setCameraManipulator trackManip

viewer setSceneData root

if { $argc >= 1 && [lindex $argv 0] eq "-viewer" } {
    # Only use the standard OSG viewer window without any Tk widgets.
    viewer setUpViewInWindow 50 50 500 400
    viewer run
    exit 0
}

# Use the OSG viewer inside a Togl widget.
set osgwin [viewer setUpViewerAsEmbeddedInWindow 50 50 500 400]
tcl3dOsgSetOsgWin $osgwin

viewer realize

osg::Matrixd myCameraMatrix
osg::Matrixd cameraRotation
osg::Matrixd cameraTrans
cameraRotation makeRotate \
      [tcl3dDegToRad -20.0] [osg::Vec3 roll    0 1 0] \
      [tcl3dDegToRad -15.0] [osg::Vec3 pitch   1 0 0] \
      [tcl3dDegToRad  10.0] [osg::Vec3 heading 0 0 1]

# 60 meters behind and 7 meters above the tank model (calculated relatively)
cameraTrans makeTranslate 10 -50 15

myCameraMatrix mult cameraRotation cameraTrans

CreateWidgets $osgwin

PrintInfo [tcl3dOsgGetInfoString]

if { [file tail [info script]] eq [file tail $::argv0] } {
    # If started directly from tclsh or wish, then start animation.
    update
    StartAnimation
}
