# NPS09_TestKeyboard.tcl
#
# Original C++ code by Joseph Sullivan.
# See http://www.openscenegraph.org/projects/osg/wiki/Support/Tutorials
# for the original files.
#
# Modified for Tcl3D by Paul Obermeier 2009/03/20.
# See www.tcl3d.org for the Tcl3D extension.

package require tcl3d

if { ! [tcl3dHaveOsg] } {
    tk_messageBox -icon error -type ok -title "Missing Tcl3D module" \
                  -message "Demo needs the tcl3dOSG module."
    proc Cleanup {} {}
    exit 1
    return
}

# Font to be used in the Tk listbox.
set gDemo(listFont) {-family {Courier} -size 10}

# Window size.
set gDemo(winWidth)  640
set gDemo(winHeight) 480

# Determine the directory and filename of this script.
set gDemo(scriptFile) [info script]
set gDemo(scriptDir)  [file dirname $gDemo(scriptFile)]

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    puts "Error: $msg\n\n$::errorInfo"
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    ExitProg
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# Idle callback to redisplay the scene.
proc Animate {} {
    .fr.toglwin postredisplay
    set ::animateId [tcl3dAfterIdle Animate]
}

proc StartAnimation {} {
    if { ! [info exists ::animateId] } {
        Animate
    }
}

proc StopAnimation {} {
    if { [info exists ::animateId] } {
        after cancel $::animateId 
        unset ::animateId
    }
}

proc CreateCallback { toglwin } {
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    # Propagate resize event to embedded OSG window.
    tcl3dOsgWindowResize $toglwin [tcl3dOsgGetOsgWin] $w $h
}

proc DisplayCallback { toglwin } {
    if { [viewer valid] } {
        viewer frame
    }
    $toglwin swapbuffers
}

proc Cleanup {} {
    uplevel #0 unset gDemo
    viewer -delete
}

proc ExitProg {} {
    exit
}

proc SaveOsgToFile {} {
    global gDemo

    set osgRoot [viewer getSceneData]
    set outFile [format "%s.osgt" [file rootname $gDemo(scriptFile)]]
    # Create a name on the file system, if running from within a Starpack.
    set outFile [tcl3dGenExtName $outFile]
    puts "Saving scenegraph to file $outFile"

    if { ! [osgDB::writeNodeFile $osgRoot $outFile] } {
        puts "Failed to write scenegraph to file $outFile"
    }
}

proc CreateWidgets { osgwin } {
    global gDemo

    frame .fr
    pack .fr -expand 1 -fill both
    set toglwin .fr.toglwin
    togl $toglwin -width $gDemo(winWidth) -height $gDemo(winHeight) \
                  -double true -depth true -alpha true \
                  -createcommand CreateCallback \
                  -reshapecommand ReshapeCallback \
                  -displaycommand DisplayCallback 
    listbox .fr.usage -font $::gDemo(listFont) -height 4
    label   .fr.info
    grid $toglwin   -row 0 -column 0 -sticky news
    grid .fr.usage  -row 1 -column 0 -sticky news
    grid .fr.info   -row 2 -column 0 -sticky news
    grid rowconfigure .fr 0 -weight 1
    grid columnconfigure .fr 0 -weight 1
    wm title . "Tcl3D demo: Sullivan's OSG tutorial #9 (TestKeyboard)"

    wm protocol . WM_DELETE_WINDOW "ExitProg"
    bind . <Key-Escape> "ExitProg"
    bind . <Key-f>      "SaveOsgToFile"

    bind . <KeyPress-Left>     "RotateTurretLeft"
    bind . <KeyRelease-Left>   "StopTurretRotation"
    bind . <KeyPress-Right>    "RotateTurretRight"
    bind . <KeyRelease-Right>  "StopTurretRotation"

    # Propagate key and mouse events to embedded OSG window.
    bind . <KeyPress> "tcl3dOsgKeyPress $toglwin $osgwin %N"

    tcl3dOsgAddTrackballBindings $toglwin $osgwin

    .fr.usage insert end "Key-Escape     Exit"
    .fr.usage insert end "Key-Left|Right Rotate turret left|right"
    .fr.usage insert end "Key-f          Save SceneGraph to file"
    .fr.usage insert end "Mouse          Trackball"

    .fr.usage configure -state disabled
}

#
# Start of tutorial specific code.
#

# Visitor callback procedure to search for a node with a specified name.
# The found node is saved in the gTankData hash under the searched name.
proc SearchNodeCB { node args } {
    global gTankData

    set nodeName [osg::Object_getName $node]
    if { $nodeName eq $args } {
        set gTankData($args) $node
    }
}

proc UpdateTurretRotation {} {
    global gTankData

    if { ! $gTankData(doRotation) } {
        return
    }
    set gTankData(rotation) [expr {$gTankData(rotation) + $gTankData(rotDir) * 0.001}]
    osgSim::DOFTransform_setCurrentHPR $gTankData(turret) \
                        [osg::Vec3 rot $gTankData(rotation) 0.0 0.0]
}

proc UpdateGunElevation {} {
    global gTankData

    if { $gTankData(elevation) > 0.5 } {
        set gTankData(up) false
    }
    if { $gTankData(elevation) < 0.01 } {
        set gTankData(up) true
    }
    if { $gTankData(up) } {
        set gTankData(elevation) [expr {$gTankData(elevation) + 0.001}]
    } else {
        set gTankData(elevation) [expr {$gTankData(elevation) - 0.001}]
    }
    osgSim::DOFTransform_setCurrentHPR $gTankData(gun) \
                        [osg::Vec3 hpr 0.0 $gTankData(elevation) 0.0]
}

proc InitTankData { node } {
    global gTankData

    set gTankData(rotation)   0.0
    set gTankData(elevation)  0.0
    set gTankData(doRotation) false 
    set gTankData(rotDir)     1         ; # 1 = left; -1 =  right

    osg::tcl3dOsgNodeVisitor nv1 SearchNodeCB "turret"
    osg::Node_accept $node nv1

    osg::tcl3dOsgNodeVisitor nv2 SearchNodeCB "gun"
    osg::Node_accept $node nv2
}

proc StopTurretRotation {} {
    global gTankData

    set gTankData(doRotation) false
}

proc RotateTurretLeft {} {
    global gTankData

    set gTankData(doRotation) true
    set gTankData(rotDir) 1
}

proc RotateTurretRight {} {
    global gTankData

    set gTankData(doRotation) true
    set gTankData(rotDir) -1
}

proc TankDataCallback { node args } {
    UpdateTurretRotation
    UpdateGunElevation
}

osgViewer::ViewerRef viewer [osgViewer::Viewer]

osg::Group root

set tankFile [file join $::gDemo(scriptDir) "Data/Models/t72-tank/t72-tank_des.flt"]
set tankNode [osgDB::readNodeFile $tankFile]
root addChild $tankNode

InitTankData $tankNode

osg::tcl3dOsgNodeCallback tankNodeCB TankDataCallback
osg::Node_setUpdateCallback $tankNode tankNodeCB

viewer setSceneData root
        
viewer setCameraManipulator [osgGA::TrackballManipulator]

if { $argc >= 1 && [lindex $argv 0] eq "-viewer" } {
    # Only use the standard OSG viewer window without any Tk widgets.
    # In this example, the rotation via keyboard will not work without Tk.
    viewer setUpViewInWindow 50 50 500 400
    viewer run
    exit 0
}

# Use the OSG viewer inside a Togl widget.
set osgwin [viewer setUpViewerAsEmbeddedInWindow 50 50 500 400]
tcl3dOsgSetOsgWin $osgwin

viewer realize

CreateWidgets $osgwin

PrintInfo [tcl3dOsgGetInfoString]

if { [file tail [info script]] eq [file tail $::argv0] } {
    # If started directly from tclsh or wish, then start animation.
    update
    StartAnimation
}
