# earth.tcl
#
# Original C++ code by Katja Treiber and Matthias Schmidt.
# See www.cuboslocos.com for the original files.
#
# Modified for Tcl3D by Paul Obermeier 2009/08/30.
# See www.tcl3d.org for the Tcl3D extension.

package require tcl3d

if { ! [tcl3dHaveOsg] } {
    tk_messageBox -icon error -type ok -title "Missing Tcl3D module" \
                  -message "Demo needs the tcl3dOSG module."
    proc Cleanup {} {}
    exit 1
    return
}

# Font to be used in the Tk listbox.
set gDemo(listFont) {-family {Courier} -size 10}

# Window size.
set gDemo(winWidth)  640
set gDemo(winHeight) 480

# Determine the directory and filename of this script.
set gDemo(scriptFile) [info script]
set gDemo(scriptDir) [file dirname $gDemo(scriptFile)]

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    puts "Error: $msg\n\n$::errorInfo"
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    ExitProg
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

# Idle callback to redisplay the scene.
proc Animate {} {
    .fr.toglwin postredisplay
    set ::animateId [tcl3dAfterIdle Animate]
}

proc StartAnimation {} {
    if { ! [info exists ::animateId] } {
        Animate
    }
}

proc StopAnimation {} {
    if { [info exists ::animateId] } {
        after cancel $::animateId 
        unset ::animateId
    }
}

proc CreateCallback { toglwin } {
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    # Propagate resize event to embedded OSG window.
    tcl3dOsgWindowResize $toglwin [tcl3dOsgGetOsgWin] $w $h
}

proc DisplayCallback { toglwin } {
    if { [viewer valid] } {
        viewer frame
        Update [viewer elapsedTime]
    }
    $toglwin swapbuffers
}

proc Cleanup {} {
    uplevel #0 unset gDemo
    uplevel #0 unset gLightPositionUniform
    uplevel #0 unset gCloudPositionUniform
    viewer -delete
}

proc ExitProg {} {
    exit
}

proc SaveOsgToFile {} {
    global gDemo

    set osgRoot [viewer getSceneData]
    set outFile [format "%s.osgt" [file rootname $gDemo(scriptFile)]]
    # Create a name on the file system, if running from within a Starpack.
    set outFile [tcl3dGenExtName $outFile]
    puts "Saving scenegraph to file $outFile"

    if { ! [osgDB::writeNodeFile $osgRoot $outFile] } {
        puts "Failed to write scenegraph to file $outFile"
    }
}

proc CreateWidgets { osgwin } {
    global gDemo

    frame .fr
    pack .fr -expand 1 -fill both
    set toglwin .fr.toglwin
    togl $toglwin -width $gDemo(winWidth) -height $gDemo(winHeight) \
                  -double true -depth true -alpha true \
                  -createcommand CreateCallback \
                  -reshapecommand ReshapeCallback \
                  -displaycommand DisplayCallback 
    listbox .fr.usage -font $::gDemo(listFont) -height 3
    label   .fr.info
    grid $toglwin   -row 0 -column 0 -sticky news
    grid .fr.usage  -row 1 -column 0 -sticky news
    grid .fr.info   -row 2 -column 0 -sticky news
    grid rowconfigure .fr 0 -weight 1
    grid columnconfigure .fr 0 -weight 1
    wm title . "Tcl3D demo: CubosLocos tutorial PlanetEarth"

    wm protocol . WM_DELETE_WINDOW "ExitProg"
    bind . <Key-Escape> "ExitProg"
    bind . <Key-f>      "SaveOsgToFile"

    # Propagate key and mouse events to embedded OSG window.
    bind . <KeyPress> "tcl3dOsgKeyPress $toglwin $osgwin %N"

    tcl3dOsgAddTrackballBindings $toglwin $osgwin

    .fr.usage insert end "Key-Escape Exit"
    .fr.usage insert end "Key-f      Save SceneGraph to file"
    .fr.usage insert end "Mouse      Trackball"

    .fr.usage configure -state disabled
}

#
# Start of tutorial specific code.
#

proc CreateEarth {} {
    global gLightPositionUniform gCloudPositionUniform
    global gDemo

    osg::Geode sphere
    sphere addDrawable [osg::ShapeDrawable dr [osg::Sphere sp [osg::Vec3] 1]]

    set sphereStateSet [sphere getOrCreateStateSet]

    osg::Program programObject
    set vertexObject   [osg::Shader vertShader $::osg::Shader_VERTEX]
    set fragmentObject [osg::Shader fragShader $::osg::Shader_FRAGMENT]
    programObject addShader $fragmentObject
    programObject addShader $vertexObject

    set vertFile [tcl3dGetExtFile [file join $gDemo(scriptDir) "earth.vert"]]
    set fragFile [tcl3dGetExtFile [file join $gDemo(scriptDir) "earth.frag"]]
    $vertexObject   loadShaderSourceFromFile $vertFile
    $fragmentObject loadShaderSourceFromFile $fragFile

    $sphereStateSet setAttributeAndModes programObject $::osg::StateAttribute_ON
    
    # TEXTURES

    # EARTH
    set imgFile [tcl3dGetExtFile [file join $gDemo(scriptDir) "Data/EarthMap_2500x1250.rgb"]]
    set img [osgDB::readImageFile $imgFile]
    if { $img eq "NULL" } {
        puts "Error reading texture $imgFile"
        exit 1
    }
    
    osg::Texture2D texture
    texture setDataVariance $::osg::Object_DYNAMIC
    texture setFilter $::osg::Texture_MIN_FILTER $::osg::Texture_LINEAR_MIPMAP_LINEAR
    texture setFilter $::osg::Texture_MAG_FILTER $::osg::Texture_LINEAR
    texture setWrap $::osg::Texture_WRAP_S $::osg::Texture_REPEAT
    texture setWrap $::osg::Texture_WRAP_T $::osg::Texture_CLAMP
    texture setImage $img

    # use texture unit 0
    $sphereStateSet setTextureAttributeAndModes 0 texture $::osg::StateAttribute_ON

    # use texture unit 0
    $sphereStateSet addUniform [osg::Uniform uni1 "earth_map" 0]

    # CLOUDS
    set imgFile [tcl3dGetExtFile [file join $gDemo(scriptDir) "Data/EarthClouds_2500x1250.rgb"]]
    set imgClouds [osgDB::readImageFile $imgFile]
    if { $imgClouds eq "NULL" } {
        puts "Error reading texture $imgFile"
        exit 1
    }

    osg::Texture2D textureClouds
    textureClouds setDataVariance $::osg::Object_DYNAMIC
    textureClouds setFilter $::osg::Texture_MIN_FILTER $::osg::Texture_LINEAR_MIPMAP_LINEAR
    textureClouds setFilter $::osg::Texture_MAG_FILTER $::osg::Texture_LINEAR
    textureClouds setWrap $::osg::Texture_WRAP_S $::osg::Texture_REPEAT
    textureClouds setWrap $::osg::Texture_WRAP_T $::osg::Texture_CLAMP
    textureClouds setImage $imgClouds

    # use texture unit 1
    $sphereStateSet setTextureAttributeAndModes 1 textureClouds $::osg::StateAttribute_ON

    # use texture unit 0
    $sphereStateSet addUniform [osg::Uniform uni2 "earth_map_clouds" 1]

    # END TEXTURES

    set gLightPositionUniform [osg::Uniform lp "light_position" [osg::Vec4]]
    $sphereStateSet addUniform $gLightPositionUniform

    set gCloudPositionUniform [osg::Uniform cp "cloud_position" [osg::Vec2]]
    $sphereStateSet addUniform $gCloudPositionUniform

    return sphere
}

proc Update { t } {
    global gLightPositionUniform gCloudPositionUniform
    set lightSpeed -0.1
    set cost [expr {cos ($t * $lightSpeed)}]
    set sint [expr {sin ($t * $lightSpeed)}]
    $gLightPositionUniform set [osg::Vec4 lpos $cost $sint 0.0 0.0]
    $gCloudPositionUniform set [osg::Vec2 cpos [expr {$t/500.0}] 0.0]
}

# Create the viewer and the Tk widgets.
osgViewer::ViewerRef viewer [osgViewer::Viewer]

# set the scene-graph data the viewer will render
osg::Group scene
scene addChild [CreateEarth]
viewer setSceneData scene

viewer setCameraManipulator [osgGA::TrackballManipulator]

if { $argc >= 1 && [lindex $argv 0] eq "-viewer" } {
    # Only use the standard OSG viewer window without any Tk widgets.
    viewer setUpViewInWindow 50 50 500 400
    viewer realize
    while { ! [viewer done] } {
        viewer frame
        Update [viewer elapsedTime]
    }
    exit 0
}

# Use the OSG viewer inside a Togl widget.
set osgwin [viewer setUpViewerAsEmbeddedInWindow 50 50 500 400]
tcl3dOsgSetOsgWin $osgwin

viewer realize

CreateWidgets $osgwin

PrintInfo [tcl3dOsgGetInfoString]

if { [file tail [info script]] eq [file tail $::argv0] } {
    # If started directly from tclsh or wish, then start animation.
    update
    StartAnimation
}
