// Mandelbrot shader using GPGPU techniques
// Author: Gabriel Zachmann, June 2007
//
// Modified and extended for Tcl3D by Paul Obermeier 2009/01/04
// See www.tcl3d.org for the Tcl3D extension.
//
// This shader computes nice colors from the iteration count
// stored in each texel using 'renormalized iteration count'.

#version 110

uniform sampler2D zn;         // final texture containing z_n (or earlier z_i)

uniform float MaxIterations;  // max. num. of iterations
uniform float ColorMethod;    // Coloring method
uniform vec4 BackColor;       // Color, if outside of iteration bound
uniform vec4 InnerColor;      // two colors that the bands around (try: red)
uniform vec4 OuterColor;      // the mandelbrot set interpolate (try: blue)
uniform float BandFrequ;      // try: 0.1 .. 0.01

void main()
{
    vec4 inputValue = texture2D( zn, gl_TexCoord[0].xy );
    vec2 z = inputValue.xy;
    float n = inputValue.z;

    if ( n  >= MaxIterations ) {
        gl_FragColor = BackColor;
        return;
    }
    if ( ColorMethod == 0.0 ) {
        // Coloring method Random
        gl_FragColor = (vec4 (noise3 (inputValue.xyz), 1.0) + vec4 (1.0, 1.0, 1.0, 0.0)) / 2.0;
    } else {
        // coloring using 'renormalized iteration count'
        // (see e.g. http://linas.org/art-gallery/escape/smooth.html )
        float f = n + 1.0 - log(log(length(z))) / log (2.0);
        gl_FragColor = mix (InnerColor, OuterColor, f * BandFrequ);
    }
}
