# Copyright:      2005-2025 Paul Obermeier (obermeier@tcl3d.org)
#
#                 See the file "Tcl3D_License.txt" for information on 
#                 usage and redistribution of this file, and for a
#                 DISCLAIMER OF ALL WARRANTIES.
#
# Module:         Tcl3D -> tcl3dOgl
# Filename:       gluCylinder.tcl
#
# Author:         Paul Obermeier
#
# Description:    Tcl3D demo showing the use of gluQuadric routines
#                 to draw a cylinder.

package require Tk
package require tcl3d

# Font to be used in the Tk listbox.
set listFont {-family {Courier} -size 10}

# Show errors occuring in the Togl callbacks.
proc bgerror { msg } {
    tk_messageBox -icon error -type ok -message "Error: $msg\n\n$::errorInfo"
    exit
}

# Print info message into widget a the bottom of the window.
proc PrintInfo { msg } {
    if { [winfo exists .fr.info] } {
        .fr.info configure -text $msg
    }
}

proc CreateCallback { toglwin } {
    set mat_ambient { 0.5 0.5 0.5 1.0 }
    set mat_specular { 1.0 1.0 1.0 1.0 }
    set mat_shininess { 50.0 }
    set light_position { 1.0 1.0 1.0 0.0 }
    set model_ambient { 0.5 0.5 0.5 1.0 }

    glClearColor 0.0 0.0 0.0 0.0

    glMaterialfv GL_FRONT GL_AMBIENT $mat_ambient
    glMaterialfv GL_FRONT GL_SPECULAR $mat_specular
    glMaterialfv GL_FRONT GL_SHININESS $mat_shininess
    glLightfv GL_LIGHT0 GL_POSITION $light_position
    glLightModelfv GL_LIGHT_MODEL_AMBIENT $model_ambient

    glEnable GL_LIGHTING
    glEnable GL_LIGHT0
    glEnable GL_DEPTH_TEST
}

proc DisplayCallback { toglwin } {
    glClear [expr $::GL_COLOR_BUFFER_BIT | $::GL_DEPTH_BUFFER_BIT]

    # Viewport command is not really needed, but has been inserted for
    # Mac OSX. Presentation framework (Tk) does not send a reshape event,
    # when switching from one demo to another.
    glViewport 0 0 [$toglwin width] [$toglwin height]

    glEnable GL_LIGHTING
    glShadeModel GL_SMOOTH
    glLoadIdentity
    glRotatef 45 1 0 0

    set qobj  [gluNewQuadric]
    gluQuadricOrientation $qobj GLU_OUTSIDE
    gluQuadricDrawStyle $qobj GLU_FILL 
    gluQuadricNormals $qobj GLU_FLAT
    gluCylinder $qobj 0.5 0.5 1.0 15 5

    glFlush
    $toglwin swapbuffers
}

proc ReshapeCallback { toglwin { w -1 } { h -1 } } {
    set w [$toglwin width]
    set h [$toglwin height]

    glViewport 0 0 $w $h
    glMatrixMode GL_PROJECTION
    glLoadIdentity
    if { $w <= $h } {
        glOrtho -2.5 2.5 \
              [expr -2.5*double($h)/double($w)] \
              [expr 2.5*double($h)/double($w)] \
              -10.0 10.0
    } else {
        glOrtho [expr -2.5*double($w)/double($h)] \
              [expr  2.5*double($w)/double($h)] \
              -2.5 2.5 -10.0 10.0
    }
    glMatrixMode GL_MODELVIEW
    glLoadIdentity
}

frame .fr
pack .fr -expand 1 -fill both
togl .fr.toglwin -width 500 -height 500 \
                 -double true -depth true \
                 -createcommand CreateCallback \
                 -reshapecommand ReshapeCallback \
                 -displaycommand DisplayCallback 
listbox .fr.usage -height 1 -font $listFont
label   .fr.info
grid .fr.toglwin -row 0 -column 0 -sticky news
grid .fr.usage   -row 1 -column 0 -sticky news
grid .fr.info    -row 2 -column 0 -sticky news
grid rowconfigure .fr 0 -weight 1
grid columnconfigure .fr 0 -weight 1
wm title . "Tcl3D demo: Cylinder with gluQuadric"

bind . <Key-Escape> "exit"
.fr.usage insert end "Key-Escape Exit"
.fr.usage configure -state disabled

PrintInfo [tcl3dOglGetInfoString]
