/******************************************************************************
 *{@C
 *      Copyright:      2009-2025 Paul Obermeier (obermeier@tcl3d.org)
 *
 *                      See the file "Tcl3D_License.txt" for information on
 *                      usage and redistribution of this file, and for a
 *                      DISCLAIMER OF ALL WARRANTIES.
 *
 *      Module:         Tcl3D -> tcl3dOsg
 *      Filename:       tcl3dOsgNodeCallback.i
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    SWIG file for wrapping the Open Scene Graph library.
 *                      This file is responsible to supply Tcl callback
 *                      functions to be used as OSG node callbacks.
 *
 *                      The macro USE_OSG_IF must be defined at compilation
 *                      time to include the functionality of the OSG library
 *                      into the Tcl3D package.
 *
 *****************************************************************************/

%include tclinterp.i

%{
    #include <osg/Callback>
    #include <osg/NodeVisitor>
    #include <osg/Node>

    #include <iostream>
    #include <vector>

    class tcl3dOsgNodeCallback : public osg::NodeCallback {
    public: 

        tcl3dOsgNodeCallback (Tcl_Interp *interp);

        tcl3dOsgNodeCallback (Tcl_Interp *interp, const std::string &tclProc,
                              const std::string &tclArgs="");

        virtual void operator() (osg::Node* node, osg::NodeVisitor* nv);

        void setCallbackProc (const std::string &tclProc,
                              const std::string &tclArgs="");

        const std::string& getCallbackProc ();

        const std::string& getCallbackArgs ();

    private: 
        std::string _tclProc;
        std::string _tclArgs;
        Tcl_Interp  *_interp;
    };
%}

class tcl3dOsgNodeCallback : public osg::NodeCallback {
public: 

    tcl3dOsgNodeCallback (Tcl_Interp *interp);

    tcl3dOsgNodeCallback (Tcl_Interp *interp, const std::string &tclProc,
                          const std::string &tclArgs="");

    virtual void operator() (osg::Node* node, osg::NodeVisitor* nv);

    void setCallbackProc (const std::string &tclProc, const std::string &tclArgs = "");

    const std::string& getCallbackProc ();

    const std::string& getCallbackArgs ();

private:
    std::string _tclProc;
    std::string _tclArgs;
    Tcl_Interp  *_interp;
};

%{

tcl3dOsgNodeCallback::tcl3dOsgNodeCallback (Tcl_Interp *interp) :
    osg::NodeCallback (), 
    _tclProc ("unknown"),
    _tclArgs (""),
    _interp (interp) 
{
#ifdef DEBUG_NODE_CALLBACK
    printf ("tcl3dOsgNodeCallback ()\n"); fflush (stdout);
#endif
}

tcl3dOsgNodeCallback::tcl3dOsgNodeCallback (Tcl_Interp *interp, 
                                            const std::string& tclProc,
                                            const std::string& tclArgs) : 
    osg::NodeCallback (),
    _tclProc (tclProc),
    _tclArgs (tclArgs),
    _interp (interp)
{
#ifdef DEBUG_NODE_CALLBACK
    printf ("tcl3dOsgNodeCallback (tclProc=%s tclArgs=%s)\n", 
            tclProc.c_str(), tclArgs.c_str()); fflush (stdout);
#endif
}

void tcl3dOsgNodeCallback::operator() (osg::Node* node, osg::NodeVisitor* nv)
{
    char cmd[256];
    char buf[30];
    char *b = buf;

    memset (buf, 0, sizeof (buf));

#ifdef DEBUG_NODE_CALLBACK
    printf ("tcl3dOsgNodeCallback::op() (node=%s)\n", node->getName().c_str());
    fflush (stdout);
#endif

    /* OPA TODO
     * This way of creating a SWIG compatible pointer is a hack and should
     * be improved.
     */
    b = SWIG_PackData (b, &node, sizeof (void *));
    sprintf (cmd, "%.*s _%s_p_%s__%s \"%s\"",
             200, _tclProc.c_str(), buf, node->libraryName(), node->className(),
             _tclArgs.c_str());
    if (Tcl_Eval (_interp, cmd) != TCL_OK) {
        Tcl_BackgroundError (_interp);
    }
    traverse(node, nv); 
}

void tcl3dOsgNodeCallback::setCallbackProc (const std::string &tclProc, 
                                            const std::string &tclArgs)
{
#ifdef DEBUG_NODE_CALLBACK
    printf ("tcl3dOsgNodeCallback::setCallbackProc (tclProc=%s tclArgs=%s)\n",
            tclProc.c_str(), tclArgs.c_str()); fflush (stdout);
#endif
    _tclProc = tclProc;
    _tclArgs = tclArgs;
}

const std::string& tcl3dOsgNodeCallback::getCallbackProc ()
{
    return _tclProc;
}

const std::string& tcl3dOsgNodeCallback::getCallbackArgs ()
{
    return _tclArgs;
}
%}
