/******************************************************************************
 *{@C
 *      Copyright:      2009-2025 Paul Obermeier (obermeier@tcl3d.org)
 *
 *                      See the file "Tcl3D_License.txt" for information on
 *                      usage and redistribution of this file, and for a
 *                      DISCLAIMER OF ALL WARRANTIES.
 *
 *      Module:         Tcl3D -> tcl3dOsg
 *      Filename:       osgBasic.i
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    SWIG file for wrapping the Open Scene Graph library.
 *                      This file does the actual wrapping of some of the
 *                      basic OSG module classes.
 *
 *****************************************************************************/

/* SWIG wrapper file for the following base OSG header files:
 * All vector and vector array related files (Vec2?, Vec3?, Vec4?, Array).
 * Matrix and Quaternions functions (Matrix?, Quat).
 * BoundingBox and BoundingSphere.
 * Plane.
 */ 

/* This is a little macro trick to prevent a swig error. */
#define X_AXIS(a,b,c) X_AXIS=Vec3f(1.0,0.0,0.0);
#define Y_AXIS(a,b,c) Y_AXIS=Vec3f(0.0,1.0,0.0);
#define Z_AXIS(a,b,c) Z_AXIS=Vec3f(0.0,0.0,1.0);

/* Define a SWIG macro to ignore all operators and methods
 * not wrapable by SWIG. Some of these are redefined as methods
 * via the SWIG rename functionality. See OPERATOR_RENAME macro
 * defined in file tcl3dSwigDefines.i.
 */
%define %VEC_IGNORE(NAME)
    %ignore osg::##NAME::operator[];   // Redefined as method get.
    %ignore osg::##NAME::operator();   // Redefined as method get.
    %ignore osg::##NAME::operator=;    // Redefined as method copy.
    %ignore osg::##NAME::x();
    %ignore osg::##NAME::y();
    %ignore osg::##NAME::z();
    %ignore osg::##NAME::w();
    %ignore osg::##NAME::r();
    %ignore osg::##NAME::g();
    %ignore osg::##NAME::b();
    %ignore osg::##NAME::a();
%enddef

/* Ignore the operators converting a double precision vector
 * into a single precision vector. These are available in Tcl
 * as asVec2f, asVec3f and asVec4f methods.
 */
%ignore osg::Vec2d::operator Vec2f() const;
%ignore osg::Vec3d::operator Vec3f() const;
%ignore osg::Vec4d::operator Vec4f() const;

/* Define basic SWIG extend macro usable for all OSG vector classes. */
%define %VEC_EXTEND(NAME)
%extend osg::##NAME {

    // Return size of the vector.
    int size () const {
        return static_cast<int> (osg::##NAME::num_components);
    }

    // Assign a vector to another vector, i.e. copy it.
    NAME copy () const {
        return *self;
    }

    // Return element i of the vector.
    value_type get (int i) const {
        return (*self)[i];
    }

    // Set element i of the vector.
    void setElem (int i, value_type rhs) {
        (*self)[i] = rhs;
    }

    %rename(cross) operator ^;

};
%enddef

/* Define SWIG extend macro usable for all OSG floating-point vector classes. */
%define %VECF_EXTEND(NAME)
%extend osg::##NAME {
    // Negate vector.
    const NAME negate () const {
        return -*self;
    }

    // Return the dotproduct of two vectors.
    value_type dot (const NAME &rhs) const {
        return *self * rhs;
    }
};
%enddef

/* Define a SWIG macro to extend the OSG vector classes with
 * methods for casting a double precision value to a single
 * precision vector. 
 */
%define %VEC_EXTEND_CASTS(FROM,TO)
%extend osg::##FROM {
    TO as##TO () const {
        return TO(*self);
    }
};
%enddef

/* Call the ignore SWIG macro for all OSG vector types. */
%VEC_IGNORE(Vec2b)
%VEC_IGNORE(Vec2s)
%VEC_IGNORE(Vec2f)
%VEC_IGNORE(Vec2d)

%VEC_IGNORE(Vec3b)
%VEC_IGNORE(Vec3s)
%VEC_IGNORE(Vec3f)
%VEC_IGNORE(Vec3d)

%VEC_IGNORE(Vec4ub)
%VEC_IGNORE(Vec4b)
%VEC_IGNORE(Vec4s)
%VEC_IGNORE(Vec4f)
%VEC_IGNORE(Vec4d)

/* Call the rename SWIG macro for all OSG vector types. */
%OPERATOR_RENAME(osg::Vec2b)
%OPERATOR_RENAME(osg::Vec2s)
%OPERATOR_RENAME(osg::Vec2f)
%OPERATOR_RENAME(osg::Vec2d)

%OPERATOR_RENAME(osg::Vec3b)
%OPERATOR_RENAME(osg::Vec3s)
%OPERATOR_RENAME(osg::Vec3f)
%OPERATOR_RENAME(osg::Vec3d)

%OPERATOR_RENAME(osg::Vec4ub)
%OPERATOR_RENAME(osg::Vec4b)
%OPERATOR_RENAME(osg::Vec4s)
%OPERATOR_RENAME(osg::Vec4f)
%OPERATOR_RENAME(osg::Vec4d)

/* Call the basic SWIG extend macro for all OSG vector types. */
%VEC_EXTEND(Vec2b)
%VEC_EXTEND(Vec2s)
%VEC_EXTEND(Vec2f)
%VEC_EXTEND(Vec2d)

%VEC_EXTEND(Vec3b)
%VEC_EXTEND(Vec3s)
%VEC_EXTEND(Vec3f)
%VEC_EXTEND(Vec3d)

%VEC_EXTEND(Vec4ub)
%VEC_EXTEND(Vec4b)
%VEC_EXTEND(Vec4s)
%VEC_EXTEND(Vec4f)
%VEC_EXTEND(Vec4d)

/* Call the SWIG extend macro for all floating point vector types. */
%VECF_EXTEND (Vec2f)
%VECF_EXTEND (Vec3f)
%VECF_EXTEND (Vec4f)
%VECF_EXTEND (Vec2d)
%VECF_EXTEND (Vec3d)
%VECF_EXTEND (Vec4d)

/* Call the SWIG extend macro for all vector types supporting down-casts. */
%VEC_EXTEND_CASTS (Vec2d,Vec2f)
%VEC_EXTEND_CASTS (Vec3d,Vec3f)
%VEC_EXTEND_CASTS (Vec4d,Vec4f)

/* Wrap the OSG vector classes. */
%include osg/Vec2b
%include osg/Vec3b
%include osg/Vec4b

%include osg/Vec4ub

%include osg/Vec2s
%include osg/Vec3s
%include osg/Vec4s

%include osg/Vec2f
%include osg/Vec3f
%include osg/Vec4f

%include osg/Vec2d
%include osg/Vec3d
%include osg/Vec4d

%include osg/Vec2
%include osg/Vec3
%include osg/Vec4

/* Wrap the Array template classes. */
/* OPA TODO The following defines must be taken from the OpenGL headers. */
#define GL_FLOAT  0x1406
#define GL_DOUBLE 0x140A
#define GL_UNSIGNED_INT 0x1405
typedef unsigned char GLubyte;
typedef unsigned short GLushort;
#ifdef Darwin_10_4
typedef unsigned long GLuint;
typedef long GLint;
typedef long GLsizei;
%template(Stduint) std::vector<unsigned int>;
#else
typedef unsigned int GLuint;
typedef int GLint;
typedef int GLsizei;
#endif

%include osg/BufferObject
%include osg/Array

%template(StdVec2f) std::vector<osg::Vec2f>;
%template(StdVec3f) std::vector<osg::Vec3f>;
%template(StdVec4f) std::vector<osg::Vec4f>;

%template(StdVec2d) std::vector<osg::Vec2d>;
%template(StdVec3d) std::vector<osg::Vec3d>;
%template(StdVec4d) std::vector<osg::Vec4d>;

%template(MixinVec2f) osg::MixinVector<osg::Vec2f>;
%template(MixinVec3f) osg::MixinVector<osg::Vec3f>;
%template(MixinVec4f) osg::MixinVector<osg::Vec4f>;

%template(MixinVec2d) osg::MixinVector<osg::Vec2d>;
%template(MixinVec3d) osg::MixinVector<osg::Vec3d>;
%template(MixinVec4d) osg::MixinVector<osg::Vec4d>;

%template(StdGLubyte)  std::vector<GLubyte>;
%template(StdGLushort) std::vector<GLushort>;
%template(StdGLuint)   std::vector<GLuint>;
%template(StdGLsizei)  std::vector<GLsizei>;


%template(MixinGLubyte)  osg::MixinVector<GLubyte>;
%template(MixinGLushort) osg::MixinVector<GLushort>;
%template(MixinGLuint)   osg::MixinVector<GLuint>;
%template(MixinGLsizei)  osg::MixinVector<GLsizei>;

/* Do not define templates for VecXArray, because these would be overwritten
 * by the VecXfArray templates. We choose to use the names with the explicit
 * letter for the vector type. 
 * The VecXArray types are created via the "interp alias" command.
 */
%template(Vec2fArray) osg::TemplateArray<osg::Vec2f,osg::Array::Vec2ArrayType,2,GL_FLOAT>;
%template(Vec3fArray) osg::TemplateArray<osg::Vec3f,osg::Array::Vec3ArrayType,3,GL_FLOAT>;
%template(Vec4fArray) osg::TemplateArray<osg::Vec4f,osg::Array::Vec4ArrayType,4,GL_FLOAT>;

/* Note: Double arrays are not supported in OSG 2.0 */
%template(Vec2dArray) osg::TemplateArray<osg::Vec2d,osg::Array::Vec2dArrayType,2,GL_DOUBLE>;
%template(Vec3dArray) osg::TemplateArray<osg::Vec3d,osg::Array::Vec3dArrayType,3,GL_DOUBLE>;
%template(Vec4dArray) osg::TemplateArray<osg::Vec4d,osg::Array::Vec4dArrayType,4,GL_DOUBLE>;


%template(UIntArray) osg::TemplateIndexArray<GLuint,osg::Array::UIntArrayType,1,GL_UNSIGNED_INT>;

/* Wrap the Quaternion class. */

/* Define SWIG macros to extend the OSG Quaternion class with
 * methods callable by Tcl. These are additional methods or methods
 * being used instead of not wrapable operators.
 */
%define %QUAT_EXTEND(NAME)
%extend osg::##NAME {

    // Return size of the quat.
    int size () const {
        return 4;
    }

    // Assign a quat to another quat, i.e. copy it.
    NAME copy () const {
        return *self;
    }

    // Negate quat.
    const NAME negate () const {
        return -*self;
    }

    // Return element i of the quat.
    value_type get (int i) const {
        return (*self)[i];
    }

    // Set element i of the quat.
    void setElem (int i, value_type rhs) {
        (*self)[i] = rhs;
    }
};
%enddef

/* Call the ignore SWIG macro for the OSG Quat type. */
%VEC_IGNORE(Quat)

/* Call the basic SWIG rename and extend macro for the OSG quat type. */
%OPERATOR_RENAME(osg::Quat)
%QUAT_EXTEND(Quat)

/* OPA TODO getRotate conversion */
// %apply double *OUTPUT {double &angle, double &x, double &y, double &z};

%include osg/Quat

/* Wrap the Matrix classes. */

/* Define SWIG macros to extend the OSG Matrix classes with
 * methods callable by Tcl. These are additional methods or methods
 * being used instead of not wrapable operators.
 */

/* Map getOrtho and getFrustum parameters to be returned as a Tcl list. */ 
%apply double *OUTPUT {double& left,   double& right,
                       double& bottom, double& top,
                       double& zNear,  double& zFar};

/* Map getPerspective parameters to be returned as a Tcl list. */ 
%apply double *OUTPUT {double& fovy,  double& aspectRatio, 
                       double& zNear, double& zFar};

/* Define basic SWIG extend macro usable for all OSG Matrix classes. */
%define %MAT_EXTEND(NAME)
%extend osg::##NAME {

    // Return size of the matrix.
    int size () const {
        return 16;
    }

    // Assign a matrix to another matrix, i.e. copy it.
    NAME copy () {
        return *self;
    }

    // Return element (row,col) of the matrix.
    value_type get (int row, int col) const {
        return (*self)(row, col);
    }

};
%enddef

/* Call the ignore SWIG macro for the OSG Matrix types. */
%VEC_IGNORE(Matrixf)
%VEC_IGNORE(Matrixd)

/* Call the basic SWIG rename and extend macro for all OSG matrix types. */
%OPERATOR_RENAME(osg::Matrixf)
%OPERATOR_RENAME(osg::Matrixd)
%MAT_EXTEND(Matrixf)
%MAT_EXTEND(Matrixd)

%include osg/Matrixd
%include osg/Matrixf
%include osg/Matrix


/* Wrap the BoundingBox and BoundingSphere classes. */

/* Define basic SWIG extend macro usable for the OSG BV classes. */
%define %BV_EXTEND(NAME)
%extend osg::##NAME {
    // Assign a BV to another BV, i.e. copy it.
    NAME copy () const {
        return *self;
    }
    void expandBy (const osg::BoundingSphere &bs) {
        self->expandBy (bs);
    }
};
%enddef

%define %BS_EXTEND(NAME)
%extend osg::##NAME {
    // Extend the templated extendBy methods.
    void expandBy (const osg::Vec3f &v) {
        self->expandBy (v);
    }
    void expandBy (const osg::Vec3d &v) {
        self->expandBy (v);
    }
    void expandBy (const osg::BoundingBox &bb) {
        self->expandBy (bb);
    }
    void expandRadiusBy (const osg::Vec3f &v) {
        self->expandRadiusBy (v);
    }
    void expandRadiusBy (const osg::Vec3d &v) {
        self->expandRadiusBy (v);
    }
    void expandRadiusBy (const osg::BoundingBox &bb) {
        self->expandRadiusBy (bb);
    }
};
%enddef

%ignore osg::BoundingBoxImpl::xMin();
%ignore osg::BoundingBoxImpl::xMax();
%ignore osg::BoundingBoxImpl::yMin();
%ignore osg::BoundingBoxImpl::yMax();
%ignore osg::BoundingBoxImpl::zMin();
%ignore osg::BoundingBoxImpl::zMax();

%ignore osg::BoundingSphereImpl::radius();

%BV_EXTEND(BoundingBoxImpl)
%BV_EXTEND(BoundingSphereImpl)
%BS_EXTEND(BoundingSphereImpl)

%include osg/BoundingBox
%include osg/BoundingSphere

#ifdef OSG_USE_FLOAT_BOUNDINGSPHERE
    %template(BoundingSpheref) osg::BoundingSphereImpl<osg::Vec3f>;
#else
    %template(BoundingSphered) osg::BoundingSphereImpl<osg::Vec3d>;
#endif

#ifdef OSG_USE_FLOAT_BOUNDINGBOX
    %template(BoundingBoxf) osg::BoundingBoxImpl<osg::Vec3f>;
#else
    %template(BoundingBoxd) osg::BoundingBoxImpl<osg::Vec3d>;
#endif

/* Wrap the Plane class. */

/* Define basic SWIG extend macro usable for the OSG quaternion class. */
%define %PLANE_EXTEND(NAME)
%extend osg::##NAME {

    // Return size of the Plane.
    int size () const {
        return 4;
    }

    // Assign a Plane to another Plane, i.e. copy it.
    NAME copy () const {
        return *self;
    }

    // Return element i of the Plane.
    value_type get (int i) const {
        return (*self)[i];
    }

    // Set element i of the Plane.
    void setElem (int i, value_type rhs) {
        (*self)[i] = rhs;
    }

    // Rename the overloaded intersect method with Vec3d's.
    %rename(intersectd) intersect(const std::vector<Vec3d>& vertices) const;
};
%enddef

/* Call the ignore SWIG macro for the Plane class. */
%VEC_IGNORE(Plane)

/* Call the basic SWIG rename and extend macro for the Plane class. */
%OPERATOR_RENAME(osg::Plane)
%PLANE_EXTEND(Plane)

%include osg/Plane

/* Wrap the Polytope class. */

%ignore osg::Polytope::operator=;
%ignore osg::Polytope::getCurrentMask() const;

%include osg/Polytope

/* Wrap the ConvexPlanarPolygon class. */

%include osg/ConvexPlanarPolygon
