/******************************************************************************
 *{@C
 *      Copyright:      2005-2025 Paul Obermeier (obermeier@tcl3d.org)
 *
 *                      See the file "Tcl3D_License.txt" for information on
 *                      usage and redistribution of this file, and for a
 *                      DISCLAIMER OF ALL WARRANTIES.
 *
 *      Module:         Tcl3D -> tcl3dOgl
 *      Filename:       vector.i
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    SWIG file with C functions for manipulating vectors of
 *                      type tcl3dVector.
 *
 *****************************************************************************/

%{
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static int tcl3dVectorManipChannel (Tcl_Interp *interp, void *imgVector, 
                                    int width, int height, int numChans, 
                                    int chanNum, float scale, float offset)
{
    unsigned char *imgVec = (unsigned char *) imgVector;
    unsigned char *ptr, *stop;

    ptr = imgVec + chanNum;
    stop = imgVec + numChans * width * height;

    while (ptr < stop) {
        *ptr = *ptr * scale + offset;
        ptr += numChans;
    }
    return TCL_OK;
}

static int tcl3dVectorManip (Tcl_Interp *interp, void *imgVector, 
                             int width, int height, int numChans,
                             float scale, float offset)
{
    int i;
    for (i=0; i<numChans; i++) {
        tcl3dVectorManipChannel (interp, imgVector, width, height, numChans, 
                                 i, scale, offset);
    }
    return TCL_OK;
}

static int tcl3dVectorCopyChannel (Tcl_Interp *interp, 
                                   void *srcVector, void *dstVector, 
                                   int srcChan, int dstChan,
                                   int width, int height,
                                   int numSrcChans, int numDstChans)
{
    unsigned char *srcVec = (unsigned char *) srcVector;
    unsigned char *dstVec = (unsigned char *) dstVector;
    unsigned char *srcPtr, *dstPtr;
    unsigned char *stop;

    srcPtr = srcVec + srcChan;
    dstPtr = dstVec + dstChan;
    stop = srcVec + numSrcChans * width * height;

    while (srcPtr < stop) {
        *dstPtr = *srcPtr;
        srcPtr += numSrcChans;
        dstPtr += numDstChans;
    }
    return TCL_OK;
}

static int tcl3dVectorCopy (Tcl_Interp *interp,
                            void *srcVector, void *dstVector,
                            int width, int height, int numChans)
{
    int i;
    for (i=0; i<numChans; i++) {
        tcl3dVectorCopyChannel (interp, srcVector, dstVector, 
                                i, i, width, height, numChans, numChans);
    }
    return TCL_OK;
}

static int tcl3dVectorEqualChannel (Tcl_Interp *interp, 
                                    void *srcVector, void *dstVector, 
                                    int srcChan, int dstChan,
                                    int width, int height,
                                    int numSrcChans, int numDstChans)
{
    unsigned char *srcVec = (unsigned char *) srcVector;
    unsigned char *dstVec = (unsigned char *) dstVector;
    unsigned char *srcPtr, *dstPtr;
    unsigned char *stop;

    srcPtr = srcVec + srcChan;
    dstPtr = dstVec + dstChan;
    stop = srcVec + numSrcChans * width * height;

    while (srcPtr < stop) {
        if (*dstPtr != *srcPtr)
            return 0;
        srcPtr += numSrcChans;
        dstPtr += numDstChans;
    }
    return 1;
}

static int tcl3dVectorEqual (Tcl_Interp *interp,
                             void *srcVector, void *dstVector,
                             int width, int height, int numChans)
{
    int i;
    for (i=0; i<numChans; i++) {
        if (0 == tcl3dVectorEqualChannel (interp, srcVector, dstVector, 
                                          i, i, width, height, numChans, numChans))
            return 0;
    }
    return 1;
}

%}

%define %Linspace(TYPE,NAME)
%{
static int tcl3dVectorLinspace_##NAME (Tcl_Interp *interp, TYPE *vector,
                                       TYPE start, TYPE end, int n)
{
    int i;
    for (i=0; i<n; i++) {
        vector[i] = start + (end-start)*i/(n-1);
    }
    return TCL_OK;
}
%}

int tcl3dVectorLinspace_##NAME (Tcl_Interp *interp, TYPE *vector,
                                TYPE start, TYPE end, int n);
%enddef

int tcl3dVectorManipChannel (Tcl_Interp *interp, void *imgVector, 
                             int width, int height, int numChans,
                             int chanNum, float scale, float offset);
int tcl3dVectorManip   (Tcl_Interp *interp, void *imgVector, 
                        int width, int height, int numChans,
                        float scale, float offset);
int tcl3dVectorCopyChannel (Tcl_Interp *interp, 
                            void *srcVector, void *dstVector, 
                            int srcChan, int dstChan,
                            int width, int height, 
                            int numSrcChans, int numDstChans);
int tcl3dVectorCopy (Tcl_Interp *interp,
                     void *srcVector, void *dstVector,
                     int width, int height, int numChans);
int tcl3dVectorEqualChannel (Tcl_Interp *interp, 
                             void *srcVector, void *dstVector, 
                             int srcChan, int dstChan,
                             int width, int height, 
                             int numSrcChans, int numDstChans);
int tcl3dVectorEqual (Tcl_Interp *interp,
                      void *srcVector, void *dstVector,
                      int width, int height, int numChans);

%Linspace(float,float)
%Linspace(double,double)
%Linspace(GLfloat,GLfloat)
%Linspace(GLdouble,GLdouble)
%Linspace(GLubyte,GLubyte)
%Linspace(GLushort,GLushort)
%Linspace(GLuint,GLuint)
